<?php

declare(strict_types=1);

namespace Drupal\Tests\views_string_aggregation\Kernel\Query;

use Drupal\Core\Render\RenderContext;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\Tests\views\Kernel\ViewsKernelTestBase;
use Drupal\views\Tests\ViewTestData;
use Drupal\views\Views;

/**
 * Tests the core Drupal\views_string_aggregation\Plugin\views\query\VsaBase query plugin implementation.
 *
 * @group views
 */
class ViewsStringAggregationTest extends ViewsKernelTestBase
{

  use ContentTypeCreationTrait;
  use UserCreationTrait;
  use NodeCreationTrait;

  /**
   * {@inheritdoc}
   */
  public static $testViews = ['test_string_aggregation'];

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'filter',
    'views',
    'views_string_aggregation',
  ];

  /**
   * Test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node1;

  /**
   * Test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node2;

  /**
   * Test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node3;

  /**
   * Test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node4;

  /**
   * Test user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp($import_test_views = TRUE): void
  {
    parent::setUp(FALSE);
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installSchema('node', 'node_access');
    $this->installConfig(['node', 'filter', 'views', 'views_string_aggregation']);

    // Clear the views query plugin cache so that the change to the config is picked up
    // by our views_string_aggregation_views_plugins_query_alter.
    \Drupal::service('plugin.manager.views.query')->clearCachedDefinitions();

    ViewTestData::createTestViews(static::class, ['views_test_config']);
    // Create two node types.
    $this->createContentType(['type' => 'foo']);
    $this->createContentType(['type' => 'bar']);

    // Create user 1.
    $admin = $this->createUser();

    // And four nodes, two content types for testing aggregation.
    $requestTime = \Drupal::time()->getRequestTime();
    $this->node1 = $this->createNode([
      'type' => 'foo',
      'title' => 'foo1',
      'status' => 1,
      'uid' => $admin->id(),
      'created' => $requestTime - 10,
    ]);
    $this->node2 = $this->createNode([
      'type' => 'foo',
      'title' => 'foo2',
      'status' => 1,
      'uid' => $admin->id(),
      'created' => $requestTime - 5,
    ]);
    $this->node3 = $this->createNode([
      'type' => 'bar',
      'title' => 'bar1',
      'status' => 1,
      'uid' => $admin->id(),
      'created' => $requestTime,
    ]);
    $this->node4 = $this->createNode([
      'type' => 'bar',
      'title' => 'bar2',
      'status' => 1,
      'uid' => $admin->id(),
      'created' => $requestTime,
    ]);

    // Now create a user with the ability to edit bar but not foo.
    $this->testUser = $this->createUser([
      'access content overview',
      'access content',
      'edit any bar content',
      'delete any bar content',
    ]);
    // And switch to that user.
    $this->container->get('account_switcher')->switchTo($this->testUser);
  }

  /**
   * Tests that string aggregation properly groups and aggregates node titles.
   */
  public function testStringAggregationGroupsByContentType(): void
  {
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();
    $view->preExecute([]);
    $view->execute();
    // The view should have exactly 2 rows - one for each content type.
    $this->assertTrue(count($view->result) === 2, 'View should have exactly 2 rows');

    $renderer = $this->container->get('renderer');
    $title_outputs = [];
    $type_outputs = [];

    // Render each row and collect the title and type field outputs.
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
        if ($field === 'type') {
          $type_outputs[$index] = (string) $output;
        }
      }
    }

    // Sort the results by content type for predictable testing.
    $combined_results = [];
    foreach ($title_outputs as $index => $title) {
      $combined_results[] = [
        'type' => $type_outputs[$index],
        'titles' => $title,
      ];
    }
    usort($combined_results, function ($a, $b) {
      return strcmp($a['type'], $b['type']);
    });

    // First row should be 'bar' content type with aggregated titles.
    $this->assertTrue($combined_results[0]['type'] === 'bar', 'First row should be bar content type');
    $this->assertStringContainsString('bar1', $combined_results[0]['titles']);
    $this->assertStringContainsString('bar2', $combined_results[0]['titles']);
    $this->assertStringContainsString(', ', $combined_results[0]['titles']);

    // Second row should be 'foo' content type with aggregated titles.
    $this->assertTrue($combined_results[1]['type'] === 'foo', 'Second row should be foo content type');
    $this->assertStringContainsString('foo1', $combined_results[1]['titles']);
    $this->assertStringContainsString('foo2', $combined_results[1]['titles']);
    $this->assertStringContainsString(', ', $combined_results[1]['titles']);
  }

  /**
   * Tests that aggregated titles are properly comma-separated.
   */
  public function testCommaSeparatedTitles(): void
  {
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();
    $view->preExecute([]);
    $view->execute();

    $renderer = $this->container->get('renderer');
    $title_outputs = [];
    $type_outputs = [];

    // Render each row and collect the outputs.
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
        if ($field === 'type') {
          $type_outputs[$index] = (string) $output;
        }
      }
    }

    // Check that each aggregated title field contains comma-separated values.
    foreach ($title_outputs as $index => $title_output) {
      // Each row should contain comma-separated titles.
      $this->assertStringContainsString(', ', $title_output);

      // Verify the specific expected comma-separated format based on content type.
      if ($type_outputs[$index] === 'bar') {
        // Bar content should have both bar1 and bar2 titles.
        $this->assertTrue(
          strpos($title_output, 'bar1') !== FALSE && strpos($title_output, 'bar2') !== FALSE,
          'Bar content type should have both bar1 and bar2 titles'
        );
      } elseif ($type_outputs[$index] === 'foo') {
        // Foo content should have both foo1 and foo2 titles.
        $this->assertTrue(
          strpos($title_output, 'foo1') !== FALSE && strpos($title_output, 'foo2') !== FALSE,
          'Foo content type should have both foo1 and foo2 titles'
        );
      }
    }
  }

  /**
   * Tests that string aggregation respects the configured separator.
   */
  public function testConfiguredSeparator(): void
  {
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();

    // Modify the vsa_separator field on the query settings for the view
    $query_options = $view->display_handler->getOption('query');
    $query_options['options']['vsa_separator'] = ' | ';
    $view->display_handler->setOption('query', $query_options);

    $view->preExecute([]);
    $view->execute();

    $renderer = $this->container->get('renderer');
    $title_outputs = [];

    // Render each row and collect the title outputs.
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
      }
    }

    // Check that each aggregated title field uses the configured separator.
    foreach ($title_outputs as $index => $title_output) {
      $this->assertStringContainsString(' | ', $title_output);
      $this->assertStringNotContainsString(', ', $title_output);
    }
  }

  /**
   * Tests that string aggregation respects the exposed filter.
   */
  public function testExposedFilter(): void
  {
    // First test: no filter applied - should return all nodes grouped by type
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();
    $view->preExecute([]);
    $view->execute();

    // Should have 2 rows - one for each content type
    $this->assertTrue(count($view->result) === 2, 'View without filter should have 2 rows');

    // Test with filter for 'foo' titles
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();

    // Set the exposed filter input for title containing 'foo'
    $view->setExposedInput(['title' => 'foo']);

    $view->preExecute([]);
    $view->execute();

    $renderer = $this->container->get('renderer');
    $title_outputs = [];
    $type_outputs = [];

    // Render each row and collect the outputs
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
        if ($field === 'type') {
          $type_outputs[$index] = (string) $output;
        }
      }
    }

    // Should only have 1 row for 'foo' content type
    $this->assertTrue(count($view->result) === 1, 'View with foo filter should have 1 row');

    // Check that we only have foo content type
    $this->assertTrue($type_outputs[0] === 'foo', 'Filtered view should only contain foo content type');

    // Check that the titles contain both foo1 and foo2
    $this->assertStringContainsString('foo1', $title_outputs[0]);
    $this->assertStringContainsString('foo2', $title_outputs[0]);

    // Check that no bar titles are present
    $this->assertStringNotContainsString('bar1', $title_outputs[0]);
    $this->assertStringNotContainsString('bar2', $title_outputs[0]);

    // Test with filter for 'bar' titles
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();

    // Set the exposed filter input for title containing 'bar'
    $view->setExposedInput(['title' => 'bar']);

    $view->preExecute([]);
    $view->execute();

    $title_outputs = [];
    $type_outputs = [];

    // Render each row and collect the outputs
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
        if ($field === 'type') {
          $type_outputs[$index] = (string) $output;
        }
      }
    }

    // Should only have 1 row for 'bar' content type
    $this->assertTrue(count($view->result) === 1, 'View with bar filter should have 1 row');

    // Check that we only have bar content type
    $this->assertTrue($type_outputs[0] === 'bar', 'Filtered view should only contain bar content type');

    // Check that the titles contain both bar1 and bar2
    $this->assertStringContainsString('bar1', $title_outputs[0]);
    $this->assertStringContainsString('bar2', $title_outputs[0]);

    // Check that no foo titles are present
    $this->assertStringNotContainsString('foo1', $title_outputs[0]);
    $this->assertStringNotContainsString('foo2', $title_outputs[0]);

    // Test with filter for a specific title that exists in only one node
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();

    // Set the exposed filter input for title containing 'foo1'
    $view->setExposedInput(['title' => 'foo1']);

    $view->preExecute([]);
    $view->execute();

    $title_outputs = [];
    $type_outputs = [];

    // Render each row and collect the outputs
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
        if ($field === 'type') {
          $type_outputs[$index] = (string) $output;
        }
      }
    }

    // Should have 1 row for 'foo' content type (the entire aggregated row)
    $this->assertTrue(count($view->result) === 1, 'View with foo1 filter should have 1 row');

    // Check that we have foo content type
    $this->assertTrue($type_outputs[0] === 'foo', 'Filtered view should contain foo content type');

    // The aggregated row should contain ALL foo titles (foo1 AND foo2) even though we filtered on foo1
    $this->assertStringContainsString('foo1', $title_outputs[0]);
    $this->assertStringContainsString('foo2', $title_outputs[0]);

    // Test with filter for 'bar2' - should return the entire bar aggregated row
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();

    // Set the exposed filter input for title containing 'bar2'
    $view->setExposedInput(['title' => 'bar2']);

    $view->preExecute([]);
    $view->execute();

    $title_outputs = [];
    $type_outputs = [];

    // Render each row and collect the outputs
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
        if ($field === 'type') {
          $type_outputs[$index] = (string) $output;
        }
      }
    }

    // Should have 1 row for 'bar' content type (the entire aggregated row)
    $this->assertTrue(count($view->result) === 1, 'View with bar2 filter should have 1 row');

    // Check that we have bar content type
    $this->assertTrue($type_outputs[0] === 'bar', 'Filtered view should contain bar content type');

    // The aggregated row should contain ALL bar titles (bar1 AND bar2) even though we filtered on bar2
    $this->assertStringContainsString('bar1', $title_outputs[0]);
    $this->assertStringContainsString('bar2', $title_outputs[0]);
  }

  /**
   * Tests that string aggregation respects the sort order.
   */
  public function testSortOrder(): void {
    // Test ascending sort order (default)
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();
    $view->preExecute([]);
    $view->execute();

    $renderer = $this->container->get('renderer');
    $title_outputs = [];
    $type_outputs = [];

    // Render each row and collect the outputs
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
        if ($field === 'type') {
          $type_outputs[$index] = (string) $output;
        }
      }
    }

    // With ascending sort, 'bar1, bar2' should come before 'foo1, foo2' alphabetically
    $this->assertTrue(count($view->result) === 2, 'View should have 2 rows');
    $this->assertTrue($type_outputs[0] === 'bar', 'First row should be bar content type with ASC sort');
    $this->assertTrue($type_outputs[1] === 'foo', 'Second row should be foo content type with ASC sort');

    // Verify the aggregated content
    $this->assertStringContainsString('bar1', $title_outputs[0]);
    $this->assertStringContainsString('bar2', $title_outputs[0]);
    $this->assertStringContainsString('foo1', $title_outputs[1]);
    $this->assertStringContainsString('foo2', $title_outputs[1]);

    // Test descending sort order
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();

    // Change the sort order to descending
    $sorts = $view->display_handler->getOption('sorts');
    $sorts['title']['order'] = 'DESC';
    $view->display_handler->setOption('sorts', $sorts);

    $view->preExecute([]);
    $view->execute();

    $title_outputs = [];
    $type_outputs = [];

    // Render each row and collect the outputs
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
        if ($field === 'type') {
          $type_outputs[$index] = (string) $output;
        }
      }
    }

    // With descending sort, 'foo1, foo2' should come before 'bar1, bar2' alphabetically
    $this->assertTrue(count($view->result) === 2, 'View should have 2 rows');
    $this->assertTrue($type_outputs[0] === 'foo', 'First row should be foo content type with DESC sort');
    $this->assertTrue($type_outputs[1] === 'bar', 'Second row should be bar content type with DESC sort');

    // Verify the aggregated content
    $this->assertStringContainsString('foo1', $title_outputs[0]);
    $this->assertStringContainsString('foo2', $title_outputs[0]);
    $this->assertStringContainsString('bar1', $title_outputs[1]);
    $this->assertStringContainsString('bar2', $title_outputs[1]);
  }

  /**
   * Tests that contextual filter arguments work with string aggregation.
   */
  public function testContextualFilterArgument(): void {
    // Test with single argument 'bar1' - should return the bar content type row
    // since 'bar1' is contained within the aggregated string 'bar1, bar2'
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();

    // Set the contextual filter argument to a single title that should match
    $view->setArguments(['bar1']);

    $view->preExecute([]);
    $view->execute();

    $renderer = $this->container->get('renderer');
    $title_outputs = [];
    $type_outputs = [];

    // Render each row and collect the outputs
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
        if ($field === 'type') {
          $type_outputs[$index] = (string) $output;
        }
      }
    }

    // Should only have 1 row matching the bar content type
    $this->assertCount(1, $view->result, 'View with bar1 argument should have 1 row');
    $this->assertNotEmpty($type_outputs, 'Type outputs should not be empty');
    $this->assertNotEmpty($title_outputs, 'Title outputs should not be empty');

    // Check that we have bar content type
    $this->assertEquals('bar', $type_outputs[0], 'Contextual filter should return bar content type');

    // Check that the aggregated titles contain the filtered argument and both bar titles
    $this->assertStringContainsString('bar1', $title_outputs[0]);
    $this->assertStringContainsString('bar2', $title_outputs[0]);
    $this->assertStringContainsString(', ', $title_outputs[0]);

    // Test with single argument 'foo2' - should return the foo content type row
    // since 'foo2' is contained within the aggregated string 'foo1, foo2'
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();

    // Set the contextual filter argument to a single title that should match
    $view->setArguments(['foo2']);

    $view->preExecute([]);
    $view->execute();

    $title_outputs = [];
    $type_outputs = [];

    // Render each row and collect the outputs
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
        if ($field === 'type') {
          $type_outputs[$index] = (string) $output;
        }
      }
    }

    // Should only have 1 row matching the foo content type
    $this->assertCount(1, $view->result, 'View with foo2 argument should have 1 row');
    $this->assertNotEmpty($type_outputs, 'Type outputs should not be empty');
    $this->assertNotEmpty($title_outputs, 'Title outputs should not be empty');

    // Check that we have foo content type
    $this->assertEquals('foo', $type_outputs[0], 'Contextual filter should return foo content type');

    // Check that the aggregated titles contain the filtered argument and both foo titles
    $this->assertStringContainsString('foo1', $title_outputs[0]);
    $this->assertStringContainsString('foo2', $title_outputs[0]);
    $this->assertStringContainsString(', ', $title_outputs[0]);

    // Test with contextual filter argument that doesn't match any title
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();

    // Set the contextual filter argument to a value that doesn't match
    $view->setArguments(['nonexistent']);

    $view->preExecute([]);
    $view->execute();

    // Should have no results when the argument doesn't match any aggregated values
    $this->assertCount(0, $view->result, 'View with non-matching argument should have 0 rows');

    // Test with no contextual filter argument (should return all results)
    $view = Views::getView('test_string_aggregation');
    $view->setDisplay();

    // Don't set any arguments - should return all results
    $view->preExecute([]);
    $view->execute();

    // Should have 2 rows when no contextual filter is applied
    $this->assertCount(2, $view->result, 'View with no argument should have 2 rows');
  }

}
