<?php

declare(strict_types=1);

namespace Drupal\Tests\views_string_aggregation\Kernel\Query;

use Drupal\Core\Render\RenderContext;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\Tests\views\Kernel\ViewsKernelTestBase;
use Drupal\views\Tests\ViewTestData;
use Drupal\views\Views;

/**
 * Tests the core Drupal\views_string_aggregation\Plugin\views\query\VsaBase query plugin implementation.
 *
 * Using distinct aggregation to ensure no duplicates in results.
 *
 * @group views
 */
class ViewsStringAggregationDistinctTest extends ViewsKernelTestBase
{

  use ContentTypeCreationTrait;
  use UserCreationTrait;
  use NodeCreationTrait;

  /**
   * {@inheritdoc}
   */
  public static $testViews = ['test_string_aggregation_distinct'];

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'filter',
    'views',
    'views_string_aggregation',
  ];

  /**
   * Test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node1;

  /**
   * Test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node2;

  /**
   * Test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node3;

  /**
   * Test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node4;

  /**
   * Test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node5;

  /**
   * Test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $node6;

  /**
   * Test user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp($import_test_views = TRUE): void
  {
    parent::setUp(FALSE);
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installSchema('node', 'node_access');
    $this->installConfig(['node', 'filter', 'views', 'views_string_aggregation']);

    // Clear the views query plugin cache so that the change to the config is picked up
    // by our views_string_aggregation_views_plugins_query_alter.
    \Drupal::service('plugin.manager.views.query')->clearCachedDefinitions();

    ViewTestData::createTestViews(static::class, ['views_test_config']);
    // Create two node types.
    $this->createContentType(['type' => 'foo']);
    $this->createContentType(['type' => 'bar']);

    // Create user 1.
    $admin = $this->createUser();

    // And four nodes, two content types for testing aggregation.
    $requestTime = \Drupal::time()->getRequestTime();
    $this->node1 = $this->createNode([
      'type' => 'foo',
      'title' => 'foo1',
      'status' => 1,
      'uid' => $admin->id(),
      'created' => $requestTime - 10,
    ]);
    $this->node2 = $this->createNode([
      'type' => 'foo',
      'title' => 'foo2',
      'status' => 1,
      'uid' => $admin->id(),
      'created' => $requestTime - 5,
    ]);
    $this->node3 = $this->createNode([
      'type' => 'bar',
      'title' => 'bar1',
      'status' => 1,
      'uid' => $admin->id(),
      'created' => $requestTime,
    ]);
    $this->node4 = $this->createNode([
      'type' => 'bar',
      'title' => 'bar2',
      'status' => 1,
      'uid' => $admin->id(),
      'created' => $requestTime,
    ]);
    // Add some duplicate titles to ensure distinct aggregation works
    $this->node5 = $this->createNode([
      'type' => 'foo',
      'title' => 'foo1',
      'status' => 1,
      'uid' => $admin->id(),
      'created' => $requestTime,
    ]);
    $this->node6 = $this->createNode([
      'type' => 'bar',
      'title' => 'bar1',
      'status' => 1,
      'uid' => $admin->id(),
      'created' => $requestTime,
    ]);

    // Now create a user with the ability to edit bar but not foo.
    $this->testUser = $this->createUser([
      'access content overview',
      'access content',
      'edit any bar content',
      'delete any bar content',
    ]);
    // And switch to that user.
    $this->container->get('account_switcher')->switchTo($this->testUser);
  }

  /**
   * Tests that distinct string aggregation removes duplicate values.
   */
  public function testDistinctAggregation(): void
  {
    $view = Views::getView('test_string_aggregation_distinct');
    $view->setDisplay();
    $view->preExecute([]);
    $view->execute();

    $renderer = $this->container->get('renderer');
    $title_outputs = [];
    $type_outputs = [];

    // Render each row and collect the outputs.
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
        if ($field === 'type') {
          $type_outputs[$index] = (string) $output;
        }
      }
    }

    // Sort the results by content type for predictable testing.
    $combined_results = [];
    foreach ($title_outputs as $index => $title) {
      $combined_results[] = [
        'type' => $type_outputs[$index],
        'titles' => $title,
      ];
    }
    usort($combined_results, function ($a, $b) {
      return strcmp($a['type'], $b['type']);
    });

    // Test bar content type - should only have unique titles despite duplicates
    $bar_titles = $combined_results[0]['titles'];
    $this->assertTrue($combined_results[0]['type'] === 'bar', 'First row should be bar content type');

    // Check that both unique titles are present
    $this->assertStringContainsString('bar1', $bar_titles);
    $this->assertStringContainsString('bar2', $bar_titles);

    // Count occurrences of 'bar1' - should only appear once despite having two nodes with this title
    $bar1_count = substr_count($bar_titles, 'bar1');
    $this->assertTrue($bar1_count === 1, 'bar1 should appear only once due to distinct aggregation, found ' . $bar1_count . ' occurrences');

    // Test foo content type - should only have unique titles despite duplicates
    $foo_titles = $combined_results[1]['titles'];
    $this->assertTrue($combined_results[1]['type'] === 'foo', 'Second row should be foo content type');

    // Check that both unique titles are present
    $this->assertStringContainsString('foo1', $foo_titles);
    $this->assertStringContainsString('foo2', $foo_titles);

    // Count occurrences of 'foo1' - should only appear once despite having two nodes with this title
    $foo1_count = substr_count($foo_titles, 'foo1');
    $this->assertTrue($foo1_count === 1, 'foo1 should appear only once due to distinct aggregation, found ' . $foo1_count . ' occurrences');
  }

  /**
   * Tests that distinct aggregation works with different separators.
   */
  public function testDistinctAggregationWithCustomSeparator(): void
  {
    $view = Views::getView('test_string_aggregation_distinct');
    $view->setDisplay();

    // Modify the vsa_separator field on the query settings for the view
    $query_options = $view->display_handler->getOption('query');
    $query_options['options']['vsa_separator'] = ' | ';
    $view->display_handler->setOption('query', $query_options);

    $view->preExecute([]);
    $view->execute();

    $renderer = $this->container->get('renderer');
    $title_outputs = [];
    $type_outputs = [];

    // Render each row and collect the outputs.
    foreach ($view->result as $index => $row) {
      foreach (array_keys($view->field) as $field) {
        $output = $renderer->executeInRenderContext(new RenderContext(), function () use ($view, $row, $field) {
          return $view->field[$field]->advancedRender($row);
        });
        if ($field === 'title') {
          $title_outputs[$index] = (string) $output;
        }
        if ($field === 'type') {
          $type_outputs[$index] = (string) $output;
        }
      }
    }

    // Check that distinct aggregation works with custom separator
    foreach ($title_outputs as $index => $title_output) {
      // Should use the custom separator
      $this->assertStringContainsString(' | ', $title_output);
      $this->assertStringNotContainsString(', ', $title_output);

      // Test distinct functionality with custom separator
      if ($type_outputs[$index] === 'bar') {
        // Count occurrences of 'bar1' with custom separator
        $bar1_count = substr_count($title_output, 'bar1');
        $this->assertTrue($bar1_count === 1, 'bar1 should appear only once with custom separator, found ' . $bar1_count . ' occurrences');
      } elseif ($type_outputs[$index] === 'foo') {
        // Count occurrences of 'foo1' with custom separator
        $foo1_count = substr_count($title_output, 'foo1');
        $this->assertTrue($foo1_count === 1, 'foo1 should appear only once with custom separator, found ' . $foo1_count . ' occurrences');
      }
    }
  }

}
