<?php

namespace Drupal\views_string_aggregation\Plugin\views\query;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\query\Sql;

/**
 * Base for extended Views SQL plugin string aggregation support.
 */
class VsaBase extends Sql {

  /**
   * {@inheritdoc}
   */
  protected function defineOptions(): array {
    $options = parent::defineOptions();

    // Add our custom query options for string aggregation.
    $options['vsa_separator'] = [
      'default' => ',',
    ];
    $options['vsa_order_by'] = [
      'default' => '',
    ];
    $options['vsa_order_direction'] = [
      'default' => 'ASC',
    ];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);
    // Check if use aggregation is enabled on the view
    // and show the separator option if so.
    if ($this->view->getDisplay()->getOption('group_by')) {
      $form['vsa_separator'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Concatenated values separator'),
        '#default_value' => $this->options['vsa_separator'],
        '#description' => $this->t('The separator to use between string aggregated and concatenated values.'),
        '#size' => 5,
        '#maxlength' => 5,
      ];
      $field_options = $this->getFieldOptions();
      if (!empty($field_options)) {
        $form['vsa_order_by'] = [
          '#type' => 'select',
          '#title' => $this->t('Order concatenated values by'),
          '#default_value' => $this->options['vsa_order_by'] ?? '',
          '#options' => ['' => $this->t('- Default -')] + $field_options,
          '#description' => $this->t('Select a field to order the string aggregated and concatenated values by, if applicable.'),
        ];
        $form['vsa_order_direction'] = [
          '#type' => 'select',
          '#title' => $this->t('Concatenated values order direction'),
          '#default_value' => $this->options['vsa_order_direction'] ?? 'ASC',
          '#options' => [
            'ASC' => $this->t('Ascending'),
            'DESC' => $this->t('Descending'),
          ],
          '#description' => $this->t('Select the order direction for the selected field.'),
        ];
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptionsForm(&$form, FormStateInterface $form_state) {
    parent::validateOptionsForm($form, $form_state);
    $query_options = $form_state->getValue('query') ?? [];
    if (isset($query_options['options']['vsa_separator'])) {
      $separator = $form_state->getValue('query')['options']['vsa_separator'];
      // Semicolon is not allowed by default in
      // Drupal Connection::preprocessStatement.
      if ($separator && str_contains($separator, ';')) {
        $form_state->setErrorByName('vsa_separator', $this->t('Invalid separator: ; is not allowed.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getAggregationInfo(): array {
    $aggregations = parent::getAggregationInfo();

    // Add our generic string aggregation methods.
    // These methods are implemented in the database specific subclasses.
    $aggregations['string_aggregation'] = [
      'title' => $this->t('String aggregation'),
      'method' => 'vsaAggregationMethodSimple',
      'handler' => [
        'argument' => 'groupby_string',
        'filter' => 'groupby_string',
        'sort' => 'groupby_numeric',
      ],
    ];

    $aggregations['string_aggregation_distinct'] = [
      'title' => $this->t('String aggregation DISTINCT'),
      'method' => 'vsaAggregationMethodDistinct',
      'handler' => [
        'argument' => 'groupby_string',
        'filter' => 'groupby_string',
        'sort' => 'groupby_numeric',
      ],
    ];

    return $aggregations;
  }

  /**
   * Helper function to get a list of fields in the current view display.
   *
   * @return array
   *   Associative array of field options, keyed by field table.field alias.
   */
  protected function getFieldOptions(): array {
    $field_options = [];

    // Get the field handlers from the current view display.
    $field_handlers = $this->view->getDisplay()->getHandlers('field');
    // Simple field data for quick label lookup.
    $fields = $this->view->getDisplay()->getOption('fields') ?? [];

    foreach ($field_handlers as $field_id => $field_handler) {
      // Get the actual database table and field from the handler.
      $table = $field_handler->table;
      $field = $field_handler->realField;
      if (!empty($table) && !empty($field)) {
        $real_field = $table . '.' . $field;
        $label = $fields[$field_id]['label'] ?? $field_id;
        $field_options[$real_field] = $label . ' (' . $real_field . ')';
      }
    }

    return $field_options;
  }

  /**
   * Helper function to get the separator string safely SQL escaped.
   *
   * @return string
   *   Safe SQL escaped separator string.
   */
  protected function getSeparator(): string {
    return $this->getConnection()->quote($this->options['vsa_separator']);
  }

  /**
   * Helper function to get the order by clause.
   *
   * The value of vsa_order_by is already sanitized by Drupal
   * as a fixed list of field, in the query options form.
   * but we double check and validate it exists in the current view fields
   * to ensure the generated SQL is valid and safe.
   *
   * @param string|null $field
   *   Optional field to use for order by, if provided it overrides the
   *   selected field in the options, e.g required for pgsql DISTINCT.
   *
   * @return string
   *   Order by clause or empty string if none.
   */
  protected function getOrderBy(?string $field = NULL): string {
    // Use provided field override or fall back to configured option.
    $order_by_field = $field ?? $this->options['vsa_order_by'] ?? NULL;

    if (empty($order_by_field)) {
      return '';
    }

    // Validate order_by field exists in available field options.
    $field_options = $this->getFieldOptions();
    if (!isset($field_options[$order_by_field])) {
      return '';
    }

    // Validate and add order direction, from allowed options.
    $direction = in_array($this->options['vsa_order_direction'] ?? '', ['ASC', 'DESC'])
      ? $this->options['vsa_order_direction']
      : 'ASC';

    return " ORDER BY {$order_by_field} {$direction}";
  }

}
