<?php

namespace Drupal\views_string_aggregation\Plugin\views\query;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Attribute\ViewsQuery;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\views\ViewExecutable;

/**
 * MySQL/MariaDB GROUP_CONCAT aggregation support.
 *
 * Supports optionally setting the group_concat_max_len
 * for the database connection session.
 *
 * @ingroup views_query_plugins
 */
#[ViewsQuery(
  id: 'vsa_views_query_mysql',
  title: new TranslatableMarkup('SQL Query with MySQL/MariaDB GROUP_CONCAT'),
  help: new TranslatableMarkup('Query will be generated and run using the Drupal database API with MySQL/MariaDB GROUP_CONCAT support.')
)]
class MySql extends VsaBase {

  /**
   * Minimum value for group_concat_max_len is 4.
   *
   * @see https://dev.mysql.com/doc/refman/8.4/en/server-system-variables.html#sysvar_group_concat_max_len
   * @var int
   */
  protected const GROUP_CONCAT_MIN_LENGTH = 4;

  /**
   * Maximum value for group_concat_max_len is set to max 32 bit int (~4.2GB).
   *
   * @var int
   */
  protected const GROUP_CONCAT_MAX_LENGTH = 4294967295;

  /**
   * String aggregation method callback for MySql/MariaDB GROUP_CONCAT.
   */
  public function vsaAggregationMethodSimple($group_type, $field): string {
    return 'GROUP_CONCAT(' . $field . $this->getVsaOrderBy() . ' SEPARATOR ' . $this->getVsaSeparator() . ')';
  }

  /**
   * String aggregation method callback for MySql/MariaDB DISTINCT GROUP_CONCAT.
   */
  public function vsaAggregationMethodDistinct($group_type, $field): string {
    return 'GROUP_CONCAT(DISTINCT ' . $field . $this->getVsaOrderBy() . ' SEPARATOR ' . $this->getVsaSeparator() . ')';
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);
    // Add an additional options form setting for group_concat_max_len,
    // for MySQL/MariaDB only.
    $current_db_max_length = $this->getVsaDatabaseMaxLength();
    $form['vsa_max_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Concatenated values maximum length'),
      // Use the configured max length
      // or the current database setting as default.
      '#default_value' => (int) $this->options['vsa_max_length'],
      '#description' => $this->t(
        'Override the default maximum length of each concatenated value string (i.e: set session group_concat_max_len on pre-execute).
        <br />Enter "0" to use the database default of <strong>group_concat_max_len: @current</strong> bytes. (default)',
        ['@current' => $current_db_max_length]
      ),
      '#field_suffix' => $this->t('bytes'),
      '#min' => 0,
      '#max' => self::GROUP_CONCAT_MAX_LENGTH,
      '#step' => 1,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptionsForm(&$form, FormStateInterface $form_state) {
    parent::validateOptionsForm($form, $form_state);
    // Validate the max length option if set.
    $query_options = $form_state->getValue('query') ?? [];
    if (!empty($query_options['options']['vsa_max_length'])) {
      $max_length = (int) $form_state->getValue('query')['options']['vsa_max_length'];
      // Validate group_concat_max_len is within range.
      if ($max_length < self::GROUP_CONCAT_MIN_LENGTH || $max_length > self::GROUP_CONCAT_MAX_LENGTH) {
        $form_state->setErrorByName('vsa_max_length', $this->t(
          'The concatenated values maximum length (group_concat_max_len) must be between @min and @max.',
          [
            '@min' => self::GROUP_CONCAT_MIN_LENGTH,
            '@max' => self::GROUP_CONCAT_MAX_LENGTH,
          ]
        ));
      }
    }
  }

  /**
   * Get the current group_concat_max_len value from the database.
   *
   * @return int
   *   group_concat_max_len - the maximum length
   *   of the concatenated value string.
   */
  public function getVsaDatabaseMaxLength(): int {
    // Get the value of group_concat_max_len from the database variable if set.
    $max_length = $this->getConnection()->query("SHOW VARIABLES LIKE 'group_concat_max_len'")->fetchField(1);
    // Returns value or 0 if not found.
    return (int) $max_length;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(ViewExecutable $view) {
    // Set the group_concat_max_len for this session, if a max length is set.
    // We can't use bound query parameters for a SET statement,
    // so we carefully double check
    // that we have a safe integer value, within range, before setting it.
    if (!empty($this->options['vsa_max_length']) && is_numeric($this->options['vsa_max_length'])) {
      $max_length = (int) $this->options['vsa_max_length'];
      if ($max_length && ($max_length >= self::GROUP_CONCAT_MIN_LENGTH) && ($max_length <= self::GROUP_CONCAT_MAX_LENGTH)) {
        $this->getConnection()->query("SET SESSION group_concat_max_len = $max_length");
      }
    }
    // Proceed with normal execution of the view.
    parent::execute($view);
  }

}
