<?php

declare(strict_types=1);

namespace Drupal\views_themeable_text_area\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\views\Plugin\views\area\AreaPluginBase;

/**
 * Hooks for the Views Themeable Text Area module.
 */
class ViewsThemeableTextAreaHooks {

  use StringTranslationTrait;

  /**
   * Add views data definitions.
   *
   * @param array $data
   *   The views data.
   */
  #[Hook('views_data_alter')]
  public function alterViewsData(&$data): void {
    $data['views']['themeable_text_area'] = [
      'title' => $this->t('Text area (themeable)'),
      'help' => $this->t('Provide themeable markup for the area using any available text format.'),
      'area' => [
        'id' => 'themeable_text_area',
      ],
    ];

    $data['views']['themeable_text_area_custom'] = [
      'title' => $this->t('Unfiltered text (themeable)'),
      'help' => $this->t('Provide themeable markup for the area with minimal filtering.'),
      'area' => [
        'id' => 'themeable_text_area_custom',
      ],
    ];
  }

  /**
   * Set theme hooks.
   *
   * @return array
   *   The theme hooks.
   */
  #[Hook('theme')]
  public function theme(): array {
    return [
      'views_themeable_text_area' => [
        'render element' => 'area',
      ],
    ];
  }

  /**
   * Set theme suggestions for the text area.
   *
   * @param array $variables
   *   The theme variables.
   *
   * @return array
   *   Template suggestions.
   */
  #[Hook('theme_suggestions_views_themeable_text_area')]
  public function setHookSuggestions(array $variables): array {
    $suggestions = [];

    $area = $variables['area']['#area'] ?? NULL;
    if ($area instanceof AreaPluginBase) {
      $suggestions[] = 'views_themeable_text_area__' . $area->view->id();
      $suggestions[] = 'views_themeable_text_area__' . $area->view->id() . '__' . $area->areaType;
      $suggestions[] = 'views_themeable_text_area__' . $area->view->id() . '__' . $area->view->current_display;
      $suggestions[] = 'views_themeable_text_area__' . $area->view->id() . '__' . $area->view->current_display . '__' . $area->areaType;
    }

    return $suggestions;
  }

}
