<?php

namespace Drupal\views_tools\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Drupal\Core\Link;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Yaml\Yaml;
use Drupal\views\Entity\View;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Entity\Query\QueryFactory;
use Drupal\views_tools\ViewsTools;
use Drupal\Core\Form\FormBuilder;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Returns responses for Views Tools routes.
 */
class ViewsToolsController extends ControllerBase {

  /**
   * The entity query service.
   *
   * @var \Drupal\Core\Entity\Query\QueryInterface
   */
  protected $entityQuery;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The form builder service.
   *
   * @var \Drupal\Core\Form\FormBuilder
   */
  protected $formBuilder;

  /**
   * Constructs a ViewsToolsController object.
   *
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Form\FormBuilder $form_builder
   *   The form builder service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   */
  public function __construct(RendererInterface $renderer, EntityTypeManagerInterface $entity_type_manager, FormBuilder $form_builder, ConfigFactoryInterface $config_factory) {
    $this->renderer = $renderer;
    $this->entityQuery = $entity_type_manager->getStorage('view')->getQuery();
    $this->formBuilder = $form_builder;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('renderer'),
      $container->get('entity_type.manager'),
      $container->get('form_builder'),
      $container->get('config.factory')
    );
  }

  /**
   * Helper method to check if user has permission for export operations.
   *
   * @return bool
   *   TRUE if user can export, FALSE otherwise.
   */
  protected function canExport() {
    $account = $this->currentUser();
    return $account->hasPermission('export views') ||
           $account->hasPermission('administer views tools') ||
           $account->hasPermission('administer views');
  }

  /**
   * Helper method to check if user has permission for delete operations.
   *
   * @return bool
   *   TRUE if user can delete displays, FALSE otherwise.
   */
  protected function canDeleteDisplays() {
    $account = $this->currentUser();
    return $account->hasPermission('delete view displays') ||
           $account->hasPermission('administer views tools') ||
           $account->hasPermission('administer views');
  }

  /**
   * Lists all views with tools operations.
   *
   * @return array
   *   A render array for the views list.
   */
  public function viewsList() {
    $viewsIds = $this->entityQuery->execute();

    if (empty($viewsIds)) {
      return [
        '#markup' => '<p>' . $this->t('No views found.') . '</p>',
      ];
    }

    $views = View::loadMultiple($viewsIds);
    $items = [];

    foreach ($views as $view) {
      if (!$view->access('view')) {
        continue;
      }

      $dropButton = [
        '#type' => 'dropbutton',
        '#links' => [
          'list' => [
            'title' => $this->t('View Displays'),
            'url' => Url::fromRoute('views_tools.view', ['view' => $view->id()]),
          ],
        ],
      ];

      // Add export link only if user has export permission.
      if ($this->canExport()) {
        $dropButton['#links']['export'] = [
          'title' => $this->t('Export/Backup'),
          'url' => Url::fromRoute('views_tools.export', ['view' => $view->id()]),
        ];
      }
      $items[] = [
        Link::createFromRoute($view->label(), 'entity.view.edit_form', ['view' => $view->id()]),
        $this->renderer->render($dropButton),
      ];
    }

    if (empty($items)) {
      return [
        '#markup' => '<p>' . $this->t('No accessible views found.') . '</p>',
      ];
    }

    $output = [
      '#type' => 'table',
      '#header' => [$this->t('View'), $this->t('Operations')],
      '#rows' => $items,
      '#sticky' => TRUE,
    ];
    return $output;
  }

  /**
   * Displays a single view with its displays.
   *
   * @param string $view
   *   The view ID.
   *
   * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
   *   A render array or redirect response.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   When the view is not found.
   */
  public function view($view) {
    $viewEntity = View::load($view);
    if (!$viewEntity) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    if (!$viewEntity->access('view')) {
      throw new \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException();
    }

    $displays = $viewEntity->get('display');
    if (empty($displays)) {
      $this->messenger()->addWarning($this->t('This view has no displays.'));
      return $this->redirect('views_tools.list');
    }

    $displayList = [];
    foreach ($displays as $display) {
      $dropButton = [
        '#type' => 'dropbutton',
        '#links' => [
          'edit' => [
            'title' => $this->t('Edit'),
            'url' => Url::fromRoute('entity.view.edit_display_form', ['view' => $viewEntity->id(), 'display_id' => $display['id']]),
          ],
        ],
      ];

      // Add export links only if user has export permission.
      if ($this->canExport()) {
        $dropButton['#links']['export'] = [
          'title' => $this->t('Create New View'),
          'url' => Url::fromRoute('views_tools.display_export', ['view' => $viewEntity->id(), 'display_id' => $display['id']]),
        ];
        $dropButton['#links']['export_yaml'] = [
          'title' => $this->t('Export YAML'),
          'url' => Url::fromRoute('views_tools.display_export_yaml', ['view' => $viewEntity->id(), 'display_id' => $display['id']]),
        ];
      }

      // Don't allow deletion of default display and check delete permission.
      if ($display['id'] != 'default' && $this->canDeleteDisplays()) {
        $dropButton['#links']['delete'] = [
          'title' => $this->t('Delete display'),
          'url' => Url::fromRoute('views_tools.display_delete_confirm', ['view' => $viewEntity->id(), 'display_id' => $display['id']]),
        ];
      }

      $displayList[$display['id']] = [
        $display['display_title'] . ' (' . $display['id'] . ')',
        $this->renderer->render($dropButton),
      ];
    }

    $form = $this->formBuilder->getForm('\Drupal\views_tools\Form\BulkOperationForm', ['View', 'Operations'], $displayList, $viewEntity);
    return $form;
  }

  /**
   * Exports a view as YAML configuration.
   *
   * @param string $view
   *   The view ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The HTTP response with YAML content.
   */
  public function export($view) {
    $viewEntity = View::load($view);
    if (!$viewEntity) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    $configFileName = 'views.view.' . $viewEntity->id();
    $config = $this->configFactory->get($configFileName);
    $viewConfig = Yaml::dump($config->get());
    $fileName = "$configFileName.yml";

    $response = new Response($viewConfig);
    $response->headers->set('Content-Type', 'text/yaml');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $fileName . '"');

    return $response;
  }

  /**
   * Exports display as a new view.
   *
   * @param string $view
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   When the view or display is not found.
   */
  public function exportDisplayAsView($view, $display_id) {
    $viewEntity = View::load($view);
    if (!$viewEntity) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException('View not found.');
    }

    $displays = $viewEntity->get('display');
    if (!isset($displays[$display_id])) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException('Display not found.');
    }

    try {
      $newView = ViewsTools::exportDisplaysAsView($viewEntity, $display_id);
      $this->messenger()->addStatus($this->t('Display exported as new view: @label', ['@label' => $newView->label()]));
      return $this->redirect('entity.view.edit_form', ['view' => $newView->id()]);
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Error exporting display: @message', ['@message' => $e->getMessage()]));
      return $this->redirect('views_tools.view', ['view' => $view]);
    }
  }

  /**
   * Exports display as YAML.
   *
   * @param string $view
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The HTTP response with YAML content.
   */
  public function exportDisplayAsYaml($view, $display_id) {
    $viewEntity = View::load($view);
    if (!$viewEntity) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }
    return ViewsTools::exportDisplaysToYaml($viewEntity, $display_id);
  }

  /**
   * Deletes a display from a view.
   *
   * @param string $view
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   When the view or display is not found.
   */
  public function deleteDisplay($view, $display_id) {
    $viewEntity = View::load($view);
    if (!$viewEntity) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException('View not found.');
    }

    $displays = $viewEntity->get('display');
    if (!isset($displays[$display_id])) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException('Display not found.');
    }

    if ($display_id === 'default') {
      $this->messenger()->addError($this->t('Cannot delete the default display.'));
      return $this->redirect('views_tools.view', ['view' => $viewEntity->id()]);
    }

    try {
      $success = ViewsTools::deleteDisplay($viewEntity, $display_id);
      if ($success) {
        $this->messenger()->addStatus($this->t('Display @display_id has been deleted.', ['@display_id' => $display_id]));
      }
      else {
        $this->messenger()->addWarning($this->t('Display could not be deleted.'));
      }
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Error deleting display: @message', ['@message' => $e->getMessage()]));
    }

    return $this->redirect('views_tools.view', ['view' => $viewEntity->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function editDisplay($view, $display_id) {
    // @todo feature to edit a single display without affecting
    // other displays of view.
  }

}
