<?php

namespace Drupal\views_tools\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Entity\View;
use Drupal\views_tools\ViewsTools;

/**
 * Contribute form.
 */
class BulkOperationForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'views_tools_bulk_operation_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $header = NULL, $rows = NULL, $view = NULL) {
    $form['display'] = array(
      '#title' => $view->label(),
      '#type' => 'tableselect',
      '#header' => $header,
      '#options' => $rows,
      '#empty' => $this->t('No displays found'),
    );

    $form['view_id'] = [
      '#type' => 'hidden',
      '#value' => $view->id(),
    ];

    // Build operations options based on user permissions.
    $options = [];
    $account = $this->currentUser();

    // Check export permissions.
    if ($account->hasPermission('export views') ||
        $account->hasPermission('administer views tools') ||
        $account->hasPermission('administer views')) {
      $options['export'] = $this->t('Export into a single new view');
      $options['multi_export'] = $this->t('Export to YAML');
    }

    // Check delete permissions.
    if ($account->hasPermission('delete view displays') ||
        $account->hasPermission('administer views tools') ||
        $account->hasPermission('administer views')) {
      $options['delete'] = $this->t('Delete displays');
    }

    // If no operations are available, show message.
    if (empty($options)) {
      $form['no_operations'] = [
        '#markup' => '<p>' . $this->t('You do not have permission to perform any bulk operations.') . '</p>',
      ];
      return $form;
    }

    $form['display_operation'] = [
      '#title' => $this->t('Select Operation'),
      '#type' => 'select',
      '#options' => $options,
      '#required' => TRUE,
    ];
    $form['submit'] = ['#type' => 'submit', '#value' => $this->t('Submit')];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValue('display');
    $displays = array_filter($values, function ($display) {
      return !empty($display) ? $display : NULL;
    });

    if (empty($displays)) {
      $form_state->setErrorByName('display', $this->t('Please select at least one display.'));
    }

    $action = $form_state->getValue('display_operation');
    if (empty($action)) {
      $form_state->setErrorByName('display_operation', $this->t('Please select an operation.'));
    }

    // Validate that we're not trying to delete the default display.
    if ($action === 'delete' && in_array('default', $displays)) {
      $form_state->setErrorByName('display', $this->t('Cannot delete the default display.'));
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $form_state->cleanValues();
    $values = $form_state->getValue('display');
    $action = $form_state->getValue('display_operation');
    $displays = array_filter($values, function ($display) {
      return !empty($display) ? $display : NULL;
    });
    $view = View::load($form_state->getValue('view_id'));

    if (!$view) {
      $this->messenger()->addError($this->t('View not found.'));
      return;
    }

    try {
      if ($action == 'export') {
        $newView = ViewsTools::exportDisplaysAsView($view, $displays);
        $this->messenger()->addStatus($this->t('Displays exported as new view: @label', ['@label' => $newView->label()]));
        $form_state->setRedirect('entity.view.edit_form', ['view' => $newView->id()]);
      }
      elseif ($action == 'delete') {
        $success = ViewsTools::deleteDisplay($view, $displays);
        if ($success) {
          $this->messenger()->addStatus($this->t('Selected displays have been deleted.'));
        }
        else {
          $this->messenger()->addWarning($this->t('Some displays could not be deleted.'));
        }
        $form_state->setRedirect('views_tools.view', ['view' => $view->id()]);
      }
      elseif ($action == 'multi_export') {
        // For YAML export, we need to handle the response differently
        // since it returns a Response object
        $response = ViewsTools::exportDisplaysToYaml($view, $displays);
        $response->send();
        exit;
      }
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Error performing operation: @message', ['@message' => $e->getMessage()]));
      $form_state->setRedirect('views_tools.view', ['view' => $view->id()]);
    }
  }

}
