<?php

namespace Drupal\views_tools;

use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Yaml\Yaml;
use Drupal\views\Entity\View;

/**
 * Contains commonly used functions for Views Tools.
 */
class ViewsTools {

  /**
   * Deletes specified displays from a view.
   *
   * @param \Drupal\views\Entity\View $view
   *   The view entity.
   * @param array|string $display
   *   The display ID(s) to delete.
   *
   * @return bool
   *   TRUE if the view was saved successfully, FALSE otherwise.
   */
  public static function deleteDisplay(View $view, $display = []) {
    if (empty($display)) {
      return FALSE;
    }
    $displays = $view->get('display');
    foreach ($displays as $key => $item) {
      if (in_array($key, (array) $display) && $key != 'default') {
        unset($displays[$key]);
      }
    }
    $view->set('display', $displays);
    return $view->save();
  }

  /**
   * Exports specified displays as a new view.
   *
   * @param \Drupal\views\Entity\View $view
   *   The source view entity.
   * @param array|string $displayIds
   *   The display IDs to export.
   *
   * @return \Drupal\views\Entity\View
   *   The newly created view entity.
   */
  public static function exportDisplaysAsView(View $view, $displayIds = []) {
    $newViewId = $view->id();
    while (View::load($newViewId)) {
      $newViewId .= '_1';
    }
    $newView = $view->createDuplicate()->set('id', $newViewId);
    $displays = $newView->get('display');
    foreach ($displays as $key => $display) {
      if (!in_array($key, (array) $displayIds) && $key != 'default') {
        unset($displays[$key]);
      }
    }
    $newView->set('display', $displays);
    $newView->save();
    return $newView;
  }

  /**
   * Exports specified displays to YAML format.
   *
   * @param \Drupal\views\Entity\View $view
   *   The source view entity.
   * @param array|string $displayIds
   *   The display IDs to export.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The HTTP response with YAML content.
   */
  public static function exportDisplaysToYaml(View $view, $displayIds) {
    $newViewId = $view->id();
    while (View::load($newViewId)) {
      $newViewId .= '_1';
    }
    $newView = $view->createDuplicate()->set('id', $newViewId);
    $displays = $newView->get('display');
    foreach ($displays as $key => $display) {
      if (!in_array($key, (array) $displayIds) && $key != 'default') {
        unset($displays[$key]);
      }
    }
    $newView->set('display', $displays);
    $newView->save();
    $configFileName = 'views.view.' . $newView->id();
    $config = \Drupal::config($configFileName);
    $viewConfig = Yaml::dump($config->get());
    $newView->delete();
    $fileName = "$configFileName.yml";

    $response = new Response($viewConfig);
    $response->headers->set('Content-Type', 'text/yaml');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $fileName . '"');

    return $response;
  }

}
