<?php

namespace Drupal\Tests\views_tools\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\views\Entity\View;

/**
 * Comprehensive functional tests for Views Tools functionality.
 *
 * @group views_tools
 */
class ViewsToolsComprehensiveTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['views_tools', 'views', 'views_ui', 'node'];

  /**
   * A user with permission to administer views.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A test view entity.
   *
   * @var \Drupal\views\Entity\View
   */
  protected $testView;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'administer views',
      'access administration pages',
    ]);

    // Create a test view with multiple displays.
    $this->createTestView();
  }

  /**
   * Creates a test view with multiple displays.
   */
  protected function createTestView() {
    $this->testView = View::create([
      'id' => 'test_views_tools_view',
      'label' => 'Test Views Tools View',
      'module' => 'views',
      'description' => '',
      'tag' => '',
      'base_table' => 'node_field_data',
      'base_field' => 'nid',
      'display' => [
        'default' => [
          'id' => 'default',
          'display_title' => 'Default',
          'display_plugin' => 'default',
          'position' => 0,
          'display_options' => [],
        ],
        'page_1' => [
          'id' => 'page_1',
          'display_title' => 'Page 1',
          'display_plugin' => 'page',
          'position' => 1,
          'display_options' => [
            'path' => 'test-page-1',
          ],
        ],
        'page_2' => [
          'id' => 'page_2',
          'display_title' => 'Page 2',
          'display_plugin' => 'page',
          'position' => 2,
          'display_options' => [
            'path' => 'test-page-2',
          ],
        ],
      ],
    ]);
    $this->testView->save();
  }

  /**
   * Tests the views tools list page.
   */
  public function testViewsToolsList() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/views-tools');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Test Views Tools View');
    $this->assertSession()->linkExists('View Displays');
    $this->assertSession()->linkExists('Export/Backup');
  }

  /**
   * Tests the individual view displays page.
   */
  public function testViewDisplaysPage() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/views-tools/view/test_views_tools_view');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Default (default)');
    $this->assertSession()->pageTextContains('Page 1 (page_1)');
    $this->assertSession()->pageTextContains('Page 2 (page_2)');

    // Check that the default display doesn't have delete option.
    $this->assertSession()->elementNotExists('css', 'a[href*="delete/test_views_tools_view/display/default/confirm"]');

    // Check that other displays have delete option.
    $this->assertSession()->elementExists('css', 'a[href*="delete/test_views_tools_view/display/page_1/confirm"]');
  }

  /**
   * Tests view export functionality.
   */
  public function testViewExport() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/views-tools/export/test_views_tools_view');

    // Check that we get a YAML response.
    $this->assertSession()->responseHeaderContains('Content-Type', 'text/yaml');
    $this->assertSession()->responseHeaderContains('Content-Disposition', 'attachment');
    $this->assertSession()->responseHeaderContains('filename', 'views.view.test_views_tools_view.yml');
  }

  /**
   * Tests display export as new view functionality.
   */
  public function testDisplayExportAsView() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/views-tools/export/test_views_tools_view/display/page_1');

    // Should redirect to edit form of new view.
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->addressMatches('/\/admin\/structure\/views\/view\/test_views_tools_view_\d+/');
  }

  /**
   * Tests display deletion confirmation form.
   */
  public function testDisplayDeleteConfirmation() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/views-tools/delete/test_views_tools_view/display/page_1/confirm');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Do you want to delete display page_1?');
    $this->assertSession()->pageTextContains('This action cannot be undone!');
    $this->assertSession()->buttonExists('Delete it Now!');
    $this->assertSession()->linkExists('Cancel');
  }

  /**
   * Tests display deletion functionality.
   */
  public function testDisplayDeletion() {
    $this->drupalLogin($this->adminUser);

    // Confirm deletion.
    $this->drupalGet('/admin/structure/views-tools/delete/test_views_tools_view/display/page_1/confirm');
    $this->submitForm([], 'Delete it Now!');

    // Should redirect back to view displays page.
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->addressEquals('/admin/structure/views-tools/view/test_views_tools_view');

    // Check that the display is no longer listed.
    $this->assertSession()->pageTextNotContains('Page 1 (page_1)');
    $this->assertSession()->pageTextContains('Page 2 (page_2)');
  }

  /**
   * Tests bulk operations form.
   */
  public function testBulkOperations() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/views-tools/view/test_views_tools_view');

    // Test bulk export operation.
    $this->submitForm([
      'display[page_1]' => 'page_1',
      'display_operation' => 'export',
    ], 'Submit');

    // Should redirect to edit form of new view.
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->addressMatches('/\/admin\/structure\/views\/view\/test_views_tools_view_\d+/');
  }

  /**
   * Tests bulk delete operation.
   */
  public function testBulkDelete() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/views-tools/view/test_views_tools_view');

    // Test bulk delete operation.
    $this->submitForm([
      'display[page_1]' => 'page_1',
      'display[page_2]' => 'page_2',
      'display_operation' => 'delete',
    ], 'Submit');

    // Should redirect back to view displays page.
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->addressEquals('/admin/structure/views-tools/view/test_views_tools_view');

    // Check that the displays are no longer listed.
    $this->assertSession()->pageTextNotContains('Page 1 (page_1)');
    $this->assertSession()->pageTextNotContains('Page 2 (page_2)');
    $this->assertSession()->pageTextContains('Default (default)');
  }

  /**
   * Tests access control for views tools pages.
   */
  public function testAccessControl() {
    // Test anonymous user access.
    $this->drupalGet('/admin/structure/views-tools');
    $this->assertSession()->statusCodeEquals(403);

    // Test authenticated user without permissions.
    $regular_user = $this->drupalCreateUser();
    $this->drupalLogin($regular_user);
    $this->drupalGet('/admin/structure/views-tools');
    $this->assertSession()->statusCodeEquals(403);

    // Test user with correct permissions.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/structure/views-tools');
    $this->assertSession()->statusCodeEquals(200);
  }

}
