<?php

namespace Drupal\Tests\views_tools\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\views_tools\ViewsTools;
use Drupal\views\Entity\View;

/**
 * Unit tests for the ViewsTools class.
 *
 * @group views_tools
 * @coversDefaultClass \Drupal\views_tools\ViewsTools
 */
class ViewsToolsTest extends UnitTestCase {

  /**
   * Tests deleteDisplay method with empty display parameter.
   *
   * @covers ::deleteDisplay
   */
  public function testDeleteDisplayWithEmptyDisplay() {
    $view = $this->createMock(View::class);
    $result = ViewsTools::deleteDisplay($view, []);
    $this->assertFalse($result);
  }

  /**
   * Tests deleteDisplay method preventing deletion of default display.
   *
   * @covers ::deleteDisplay
   */
  public function testDeleteDisplayPreventDefaultDeletion() {
    $view = $this->createMock(View::class);

    $displays = [
      'default' => ['id' => 'default'],
      'page_1' => ['id' => 'page_1'],
    ];

    $view->expects($this->once())
      ->method('get')
      ->with('display')
      ->willReturn($displays);

    $view->expects($this->once())
      ->method('set')
      ->with('display', $displays); // Should not remove default

    $view->expects($this->once())
      ->method('save')
      ->willReturn(TRUE);

    $result = ViewsTools::deleteDisplay($view, 'default');
    $this->assertTrue($result);
  }

  /**
   * Tests exportDisplaysAsView method generates unique view ID.
   *
   * @covers ::exportDisplaysAsView
   */
  public function testExportDisplaysAsViewGeneratesUniqueId() {
    $originalView = $this->createMock(View::class);
    $originalView->method('id')->willReturn('test_view');

    $duplicatedView = $this->createMock(View::class);
    $duplicatedView->method('set')->willReturnSelf();
    $duplicatedView->method('save')->willReturn(TRUE);

    $originalView->expects($this->once())
      ->method('createDuplicate')
      ->willReturn($duplicatedView);

    $duplicatedView->expects($this->once())
      ->method('set')
      ->with('id', $this->stringContains('test_view'));

    // Mock View::load to simulate existing view
    $this->mockStaticMethod(View::class, 'load', function($id) {
      return $id === 'test_view' ? new \stdClass() : NULL;
    });

    ViewsTools::exportDisplaysAsView($originalView, ['page_1']);
  }

  /**
   * Mock static method helper.
   *
   * @param string $class
   *   The class name.
   * @param string $method
   *   The method name.
   * @param callable $callback
   *   The callback function.
   */
  protected function mockStaticMethod($class, $method, $callback) {
    // This is a simplified mock for demonstration
    // In real tests, you'd use proper mocking libraries
  }

}
