<?php

/**
 * @file
 * Definition of mymodule_handler_handlername.
 */

/**
 * Filters by day of the week of any date field.
 */
class views_weekday_handler_filter extends views_handler_filter_in_operator {

  /**
   * Remove the "empty" and "not empty" operators.
   *
   * @inheritDoc
   */
  public function operators(): array {
    // Remove the "empty" and "not empty" options.
    $options = parent::operators();
    unset($options['empty'], $options['not empty']);
    return $options;
  }

  /**
   * Define the options the administrator can choose from.
   *
   * @inheritDoc
   */
  public function get_value_options(): void {
    // MySQL day of week numbers.
    $this->value_options = [
      0 => t('Today (day of display)'),
      1 => t('Sunday'),
      2 => t('Monday'),
      3 => t('Tuesday'),
      4 => t('Wednesday'),
      5 => t('Thursday'),
      6 => t('Friday'),
      7 => t('Saturday'),
    ];
  }

  /**
   * Add the custom WHERE expression to the query.
   *
   * @inheritDoc
   * @noinspection PhpPossiblePolymorphicInvocationInspection
   */
  public function query(): void {
    if (empty($this->value)) {
      return;
    }
    $this->ensure_my_table();
    // We use array_values() because the checkboxes keep keys and that can cause
    // array addition problems.
    $values = array_values($this->value);
    if (in_array(0, $values)) {
      $values = array_diff($values, [0]);
      $values[] = (int) date('w') + 1;
      $values = array_values(array_unique($values));
    }
    $date_field = "$this->table_alias.$this->real_field";
    if ($this->definition['is date']) {
      $date_field = "FROM_UNIXTIME($date_field)";
    }
    if (count($values) == 1) {
      if ($this->operator == 'in') {
        $this->query->add_where_expression($this->options['group'], "DAYOFWEEK($date_field) = :value", [
          ':value' => $values[0],
        ]);
      }
      else {
        $this->query->add_where_expression($this->options['group'], "DAYOFWEEK($date_field) <> :value", [
          ':value' => $values[0],
        ]);

      }
    }
    else if ($this->operator == 'in') {
      $this->query->add_where_expression($this->options['group'], "DAYOFWEEK($date_field) IN (:values)", [
        ':values' => $values,
      ]);
    }
    else {
      $this->query->add_where_expression($this->options['group'], "DAYOFWEEK($date_field) NOT IN (:values)", [
        ':values' => $values,
      ]);
    }
  }

  /**
   * Modified summary to display all selected days, not just 8 characters.
   *
   * @inheritDoc
   */
  public function admin_summary(): string {
    if ($this->is_a_group()) {
      return t('grouped');
    }
    if (!empty($this->options['exposed'])) {
      return t('exposed');
    }
    $info = $this->operators();

    $this->get_value_options();

    if (!is_array($this->value)) {
      return '';
    }

    $operator = check_plain($info[$this->operator]['short']);
    $values = '';
    if (in_array($this->operator, $this->operator_values(1))) {
      // Remove every element which is not known.
      foreach ($this->value as $value) {
        if (!isset($this->value_options[$value])) {
          unset($this->value[$value]);
        }
      }
      // Choose different kind of output for 0, a single and multiple values.
      if (count($this->value) == 0) {
        $values = t('Unknown');
      }
      elseif (count($this->value) == 1) {
        // If any, use the 'single' short name of the operator instead.
        if (isset($info[$this->operator]['short_single'])) {
          $operator = check_plain($info[$this->operator]['short_single']);
        }

        $keys = $this->value;
        $value = array_shift($keys);
        if (isset($this->value_options[$value])) {
          $values = check_plain($this->value_options[$value]);
        }
      }
      else {
        foreach ($this->value as $value) {
          if ($values !== '') {
            $values .= ', ';
          }
          if (isset($this->value_options[$value])) {
            $values .= check_plain($this->value_options[$value]);
          }
        }
      }
    }

    return $operator . (($values !== '') ? ' ' . $values : '');
  }

}
