<?php

declare(strict_types=1);

namespace Drupal\vipps_login\Service;

use Drupal\vipps_login\ApiResponseData\AddressData;
use Drupal\vipps_login\ApiResponseData\UserData;
use GuzzleHttp\ClientInterface;
use Psr\Http\Message\ResponseInterface;

class VippsLoginManager
{
  private $configManager;

  private $stateKey;

  private $customerSession;

  private $httpClient;

  public function __construct(
    ConfigManager $configManager,
    StateKey $stateKey,
    CustomerSession $customerSession,
    ClientInterface $httpClient
  ) {
    $this->configManager = $configManager;
    $this->stateKey = $stateKey;
    $this->customerSession = $customerSession;
    $this->httpClient = $httpClient;
  }

  public function getAuthRedirect(string $scope="openid address email name phoneNumber"):string {

    $state = $this->stateKey->generate();
    $this->customerSession->setKeyState($state);

    $args = [
      'client_id' => $this->configManager->getClientId(),
      'response_type' => 'code',
      'scope' => $scope,
      'redirect_uri' => $this->configManager->getRedirectUrl(),
      'state' => $state,
    ];

    return $this->configManager->generateUrl('oauth2/auth', $args);
  }

  public function getAuthToken(string $code): string {
    $url = $this->configManager->generateUrl('oauth2/token');

    $args = [
      'grant_type' => 'authorization_code',
      'code' => $code,
      'redirect_uri' => $this->configManager->getRedirectUrl()
    ];

    $response = $this->httpClient->request('POST', $url, [
      'headers' => [
        'Authorization' => $this->configManager->getAuthorizationStringForAuthToken(),
      ],
      'form_params' => $args,
    ]);

    return $this->getResponseBody($response)->access_token;
  }

  public function getUserInfo(string $accessToken): UserData {
    $url = $this->configManager->generateUrl('userinfo');

    $response = $this->httpClient->request('GET', $url, [
      'headers' => [
        'Authorization' => $this->configManager->getAuthorizationStringForUserInfo($accessToken),
      ],
    ]);

    $responseBody = $this->getResponseBody($response);
    $addressBody = $responseBody->address[0];

    return new UserData(
      new AddressData(
        $addressBody->address_type,
        $addressBody->country,
        $addressBody->formatted,
        $addressBody->postal_code,
        $addressBody->region,
        $addressBody->street_address
      ),
      $responseBody->email_verified,
      $responseBody->email,
      $responseBody->given_name,
      $responseBody->family_name,
      $responseBody->name,
      $responseBody->phone_number,
      $responseBody->sid,
      $responseBody->sub
    );
  }

  private function getResponseBody(ResponseInterface $response):\stdClass {
    return json_decode($response->getBody()->getContents());
  }

}
