<?php

namespace Drupal\vipps_mobilepay_login\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\social_auth\Form\SocialAuthSettingsForm;
use Drupal\social_auth\Plugin\Network\NetworkInterface;
use Drupal\vipps_mobilepay\Entity\VippsMobilepayConnection;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for Vipps MobilePay Login.
 */
class VippsAuthSettingsForm extends SocialAuthSettingsForm {

  const SETTINGS = 'vipps_mobilepay_login.settings';

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'vipps_mobilepay_login_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return array_merge(
      parent::getEditableConfigNames(),
      [self::SETTINGS]
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?NetworkInterface $network = NULL): array {
    $network_id = 'vipps_mobilepay_login';
    /** @var \Drupal\social_auth\Plugin\Network\NetworkInterface $network */
    $network = $this->networkManager->createInstance($network_id);
    $form = parent::buildForm($form, $form_state, $network);

    $config = $this->config("$network_id.settings");

    $options = $this->entityTypeManager->getStorage('vipps_mobilepay_connection')->loadMultiple();
    $options = array_map(function (VippsMobilepayConnection $profile) {
      return $profile->status() ? $profile->label() : $this->t('@label (disabled)', ['@label' => $profile->label()]);
    }, $options);
    $form['network']['connection_profile'] = [
      '#type' => 'select',
      '#title' => $this->t('Vipps MobilePay Connection Profile'),
      '#required' => TRUE,
      '#description' => $this->t('Choose one of the connection profiles defined on <a href=:url>this page</a>', [':url' => Url::fromRoute('entity.vipps_mobilepay_connection.collection')->toString()]),
      '#default_value' => $config->get('connection_profile'),
      '#options' => $options,
    ];

    $form['network']['partner_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Partner mode'),
      '#default_value' => $config->get('partner_mode'),
      '#description' => $this->t('If you are a partner making API requests on behalf of a Merchant.'),
    ];

    $form['network']['show_in_login_form'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show in login form'),
      '#default_value' => $config->get('show_in_login_form'),
      '#description' => $this->t("Show button 'Login with Vipps' in the login form"),
    ];

    $form['network']['client_id']['#access'] = FALSE;
    $form['network']['client_secret']['#access'] = FALSE;

    $form['network']['authorized_redirect_url']['#weight'] = 50;
    $form['network']['advanced']['#weight'] = 999;

    $form['network']['advanced']['scopes']['#description'] =
      $this->t('Define any additional scopes to be requested, separated by a comma (e.g.: public_repo,user:follow).<br>
                The scopes \'user\' and \'user:email\' are added by default and always requested.<br>
                You can see the full list of valid scopes and their description <a href="@scopes">here</a>.', ['@scopes' => 'https://github.com/vippsas/vipps-login-api/blob/master/vipps-login-api.md#scopes']
    );

    $form['network']['advanced']['endpoints']['#description'] =
      $this->t('Define the Endpoints to be requested when user authenticates with Vipps for the first time<br>
                Enter each endpoint in different lines in the format <em>endpoint</em>|<em>name_of_endpoint</em>.<br>
                <b>For instance:</b><br>
                /user/repos|user_repos'
    );

    $form['other_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Other settings'),
      '#open' => TRUE,
      '#weight' => 3,
    ];

    $form['other_settings']['clear_cache'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Clear cache'),
      '#default_value' => $config->get('clear_cache') ?? FALSE,
      '#description' => $this->t('Clear the cache after saving configuration'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();

    $network = $form_state->get('network');
    $network_id = $network->getBaseId();

    $this->config("$network_id.settings")
      ->set('connection_profile', $values['connection_profile'])
      ->set('partner_mode', $values['partner_mode'])
      ->set('show_in_login_form', $values['show_in_login_form'])
      ->save();

    if (boolval($values['clear_cache'])) {
      drupal_flush_all_caches();
    }

    parent::submitForm($form, $form_state);
  }

}
