<?php

namespace Drupal\vipps_mobilepay_login\Plugin\Network;

use Drupal\social_auth\Plugin\Network\NetworkBase;
use Drupal\social_auth\Settings\SettingsInterface;
use Drupal\vipps_mobilepay_login\Settings\VippsAuthSettings;

/**
 * Defines a Network Plugin for Vipps MobilePay Login.
 *
 * @package Drupal\vipps_mobilepay_login\Plugin\Network
 *
 * @Network(
 *   id = "vipps_mobilepay_login",
 *   short_name = "vipps",
 *   social_network = "Vipps",
 *   img_path = "img/vipps_logo.svg",
 *   type = "social_auth",
 *   class_name = "\League\OAuth2\Client\Provider\Vipps",
 *   auth_manager = "\Drupal\vipps_mobilepay_login\VippsAuthManager",
 *   routes = {
 *     "redirect": "social_auth.network.redirect",
 *     "callback": "social_auth.network.callback",
 *     "settings_form": "social_auth.network.settings_form",
 *   },
 *   handlers = {
 *     "settings": {
 *       "class": "\Drupal\vipps_mobilepay_login\Settings\VippsAuthSettings",
 *       "config_id": "vipps_mobilepay_login.settings"
 *     }
 *   }
 * )
 */
class VippsAuth extends NetworkBase implements VippsAuthInterface {

  /**
   * {@inheritdoc}
   */
  protected function getExtraSdkSettings(): array {
    $settings = $this->settings;
    if ($settings instanceof VippsAuthSettings) {
      return [
        'merchantSerialNumber' => $settings->getMerchantSerialNumber(),
        'subscriptionKey' => $settings->getSubscriptionKey(),
        'partnerMode' => $settings->getConfig()->get('partner_mode'),
        'testMode' => $settings->getConfig()->get('test_mode'),
      ];
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  protected function validateConfig(SettingsInterface $settings): bool {
    $validate = parent::validateConfig($settings);
    if (!$validate || !$settings instanceof VippsAuthSettings) {
      return FALSE;
    }
    $merchant_serial_number = $settings->getMerchantSerialNumber();
    $subscription_key = $settings->getSubscriptionKey();
    if ($settings->getConfig()->get('partner_mode') &&
      (!$merchant_serial_number || !$subscription_key)
    ) {
      $this->loggerFactory
        ->get('vipps_mobilepay_login')
        ->error('Define Merchant Serial Number and Subscription Key on module settings.');

      return FALSE;
    }

    return TRUE;
  }

}
