<?php

namespace Drupal\vipps_mobilepay_login;

use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\social_auth\AuthManager\OAuth2Manager;
use Drupal\social_auth\User\SocialAuthUser;
use Drupal\social_auth\User\SocialAuthUserInterface;
use Drupal\vipps_mobilepay_login\Form\VippsAuthSettingsForm;
use League\OAuth2\Client\Provider\Exception\IdentityProviderException;
use League\OAuth2\Client\Token\AccessToken;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Contains all the logic for Vipps OAuth2 authentication.
 */
class VippsAuthManager extends OAuth2Manager {

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactory $configFactory
   *   Used for accessing configuration object factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   Used to get the authorization code from the callback request.
   */
  public function __construct(
    ConfigFactory $configFactory,
    LoggerChannelFactoryInterface $logger_factory,
    RequestStack $request_stack,
  ) {
    parent::__construct(
      $configFactory->get(VippsAuthSettingsForm::SETTINGS),
      $logger_factory,
      $request_stack->getCurrentRequest(),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function authenticate(): void {
    if ($code = $this->request->query->get('code')) {
      try {
        $this->setAccessToken($this->client->getAccessToken('authorization_code', ['code' => $code]));
      }
      catch (IdentityProviderException $e) {
        $this->loggerFactory->get('vipps_mobilepay_login')
          ->error('Unable to retrieve access token with error message: %message, code: %code, and response body: %response.', [
            '%message' => $e->getMessage(),
            '%code' => $e->getCode(),
            '%response' => $e->getResponseBody(),
          ]);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getUserInfo(): ?SocialAuthUserInterface {
    if (!$this->user) {
      $access_token = $this->getAccessToken();
      /** @var \League\OAuth2\Client\Provider\VippsResourceOwner $owner */
      $owner = $access_token instanceof AccessToken
        ? $this->client->getResourceOwner($access_token)
        : NULL;

      $data = $this->getExtraDetails();
      if (!is_array($data)) {
        $data = (array) $data;
      }
      $data['phone'] = $owner->getPhoneNumber();
      $data['address'] = $owner->getAddress()->toArray();
      $this->user = new SocialAuthUser(
        $owner->getName(),
        $owner->getId(),
        $this->getAccessToken(),
        $owner->getEmail(),
        $owner->getAvatarUrl(),
        $data
      );
      $this->user->setFirstName($owner->getFirstName());
      $this->user->setLastName($owner->getLastName());
    }

    return $this->user;
  }

  /**
   * {@inheritdoc}
   */
  public function getAuthorizationUrl(): string {
    $scopes = [
      'openid',
      'address',
      'email',
      'name',
      'phoneNumber',
    ];

    $extra_scopes = $this->getScopes();
    if ($extra_scopes) {
      $scopes = array_merge($scopes, explode(',', $extra_scopes));
    }

    // Returns the URL where user will be redirected.
    return $this->client->getAuthorizationUrl([
      'scope' => $scopes,
      'requested_flow' => 'automatic_return_from_vipps_app',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function requestEndPoint($method, $path, $domain = NULL, array $options = []): mixed {
    if (!$domain) {
      $domain = 'https://api.vipps.no';
    }
    $url = $domain . $path;
    $request = $this->client->getAuthenticatedRequest($method, $url, $this->getAccessToken(), $options);
    try {
      return $this->client->getParsedResponse($request);
    }
    catch (IdentityProviderException $e) {
      $this->loggerFactory->get('vipps_mobilepay_login')
        ->error('There was an error when requesting %url. Exception: %message', [
          '%url' => $url,
          '%message' => $e->getMessage(),
        ]);
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getState(): string {
    return $this->client->getState();
  }

}
