<?php

namespace Drupal\visitors\Plugin\VisitorsEvent;

use Drupal\Core\Config\Entity\ConfigEntityType;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\visitors\Attribute\VisitorsEvent;
use Drupal\visitors\VisitorsEventPluginInterface;
use Drupal\visitors\Plugin\VisitorsEventBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Plugin implementation for entity event.
 */
#[VisitorsEvent(
  id: "entity",
  label: "Entity",
  weight: -10,
)]
class Entity extends VisitorsEventBase implements VisitorsEventPluginInterface, ContainerFactoryPluginInterface {

  /**
   * Constructs a new Entity object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The route match service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly RouteMatchInterface $routeMatch,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_route_match'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function process(array $context): ?array {
    $route = $context['route'];

    if (strpos($route, 'entity.') !== 0) {
      return NULL;
    }

    $return = [];
    $entity_route = explode('.', $route);
    // Get the entity type from the route.
    $entity_type = $entity_route[1];
    $return['variables']['plugin_var_1'] = $entity_type;
    // Implode the rest of the route.
    $event = implode('.', array_slice($entity_route, 2));
    $return['event'] = $event;

    $entity = $this->routeMatch->getParameter($entity_type);
    $entity_type_definition = $this->entityTypeManager->getDefinition($entity_type);

    // Check if it's a config entity.
    $is_config_entity = $entity_type_definition instanceof ConfigEntityType;
    $is_config_entity ? 'config' : 'content';

    if ($is_config_entity) {
      $return['variables']['plugin_var_2'] = $entity?->id();
    }
    else {
      $return['variables']['plugin_var_2'] = $entity?->bundle() ?? $entity_type;
      $entity_id = $entity?->id();
      if ($entity_id) {
        $return['variables']['plugin_int_1'] = $entity_id;
      }
      if ($entity_type_definition->isRevisionable()) {
        $return['variables']['plugin_int_2'] = $entity?->getRevisionId();
      }
    }

    $forms = &drupal_static('visitors_form_alter') ?? [];

    if ($forms && count($forms) == 1) {
      $form_id = array_keys($forms)[0];
      $base_form_id = array_values($forms)[0];
      $return['variables']['plugin_var_3'] = $base_form_id ?? $form_id;
      $return['variables']['plugin_var_4'] = $form_id;
    }

    return $return;
  }

}
