<?php

declare(strict_types=1);

namespace Drupal\Tests\visitors\Kernel\Plugin;

use Drupal\KernelTests\KernelTestBase;
use Drupal\visitors\Plugin\VisitorsEventBase;

/**
 * Tests the VisitorsEventBase plugin base class.
 *
 * @group visitors
 * @coversDefaultClass \Drupal\visitors\Plugin\VisitorsEventBase
 */
class VisitorsEventBaseTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'visitors',
  ];

  /**
   * The visitors event plugin manager.
   *
   * @var \Drupal\visitors\VisitorsEventPluginManager
   */
  protected $eventPluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installSchema('system', ['sequences']);
    $this->installConfig(['user']);

    $this->eventPluginManager = $this->container->get('visitors.event_plugin_manager');
  }

  /**
   * Creates a test plugin instance.
   *
   * @param array $pluginDefinition
   *   The plugin definition.
   *
   * @return \Drupal\visitors\Plugin\VisitorsEventBase
   *   The test plugin instance.
   */
  protected function createTestPlugin(array $pluginDefinition): VisitorsEventBase {
    return new class([], 'test_event', $pluginDefinition) extends VisitorsEventBase {

      /**
       * {@inheritdoc}
       */
      public function process(array $context): ?array {
        return NULL;
      }

    };
  }

  /**
   * Tests the plugin creation and basic functionality.
   *
   * @covers ::getLabel
   * @covers ::getPlugin
   * @covers ::getWeight
   */
  public function testPluginCreation(): void {
    // Create a mock plugin definition.
    $plugin_definition = [
      'id' => 'test_event',
      'label' => 'Test Event',
      'plugin' => 'test_plugin',
      'weight' => 10,
    ];

    // Create a mock plugin instance.
    $plugin = $this->createTestPlugin($plugin_definition);

    // Test basic functionality.
    $this->assertEquals($plugin_definition['label'], $plugin->getLabel());
    $this->assertEquals($plugin_definition['plugin'], $plugin->getPlugin());
    $this->assertEquals($plugin_definition['weight'], $plugin->getWeight());
  }

  /**
   * Tests the plugin manager integration.
   */
  public function testPluginManagerIntegration(): void {
    // Get all plugin definitions.
    $definitions = $this->eventPluginManager->getDefinitions();

    // Ensure we have plugin definitions.
    $this->assertNotEmpty($definitions);

    // Test each plugin definition.
    foreach ($definitions as $definition) {
      $plugin = $this->eventPluginManager->createInstance($definition['id']);

      // Verify the plugin is an instance of VisitorsEventBase.
      $this->assertInstanceOf(VisitorsEventBase::class, $plugin);

      // Test basic functionality.
      $this->assertEquals($definition['label'], $plugin->getLabel());
      $this->assertEquals($definition['plugin'] ?? get_class($plugin), $plugin->getPlugin());
      $this->assertEquals($definition['weight'] ?? 0, $plugin->getWeight());
    }
  }

}
