<?php

declare(strict_types=1);

namespace Drupal\Tests\visitors\Unit\Plugin\views\field;

use Drupal\Tests\UnitTestCase;
use Drupal\visitors\Plugin\views\field\RefererType;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Drupal\views\ResultRow;

/**
 * Referer type field test.
 *
 * @group visitors
 * @coversDefaultClass \Drupal\visitors\Plugin\views\field\RefererType
 * @uses \Drupal\visitors\Plugin\views\field\RefererType
 */
class RefererTypeTest extends UnitTestCase {

  /**
   * The field plugin instance.
   *
   * @var \Drupal\visitors\Plugin\views\field\RefererType
   */
  protected RefererType $field;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    \Drupal::setContainer($container);

    $configuration = [];
    $plugin_id = 'visitors_referer_type';
    $plugin_definition = [];
    $this->field = RefererType::create($container, $configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Tests the create method.
   *
   * @covers ::create
   */
  public function testCreate(): void {
    $container = \Drupal::getContainer();
    $configuration = [];
    $plugin_id = 'visitors_referer_type';
    $plugin_definition = [];
    $field = RefererType::create($container, $configuration, $plugin_id, $plugin_definition);
    $this->assertInstanceOf(RefererType::class, $field);
  }

  /**
   * Tests rendering 'direct' referer type.
   *
   * @covers ::render
   */
  public function testRenderDirect(): void {
    $values = new ResultRow(['referer_type' => 'direct']);
    $this->field->field_alias = 'referer_type';

    $result = $this->field->render($values);
    $this->assertEquals('Direct Entry', (string) $result);
  }

  /**
   * Tests rendering 'website' referer type.
   *
   * @covers ::render
   */
  public function testRenderWebsite(): void {
    $values = new ResultRow(['referer_type' => 'website']);
    $this->field->field_alias = 'referer_type';

    $result = $this->field->render($values);
    $this->assertEquals('Website', (string) $result);
  }

  /**
   * Tests rendering 'internal' referer type.
   *
   * @covers ::render
   */
  public function testRenderInternal(): void {
    $values = new ResultRow(['referer_type' => 'internal']);
    $this->field->field_alias = 'referer_type';

    $result = $this->field->render($values);
    $this->assertEquals('Internal', (string) $result);
  }

  /**
   * Tests rendering unknown referer type.
   *
   * @covers ::render
   */
  public function testRenderUnknown(): void {
    $values = new ResultRow(['referer_type' => 'unknown']);
    $this->field->field_alias = 'referer_type';

    $result = $this->field->render($values);
    // Should return 'Unknown' for unknown types due to default case.
    $this->assertEquals('Unknown', (string) $result);
  }

  /**
   * Tests rendering empty referer type.
   *
   * @covers ::render
   */
  public function testRenderEmpty(): void {
    $values = new ResultRow(['referer_type' => '']);
    $this->field->field_alias = 'referer_type';

    $result = $this->field->render($values);
    // Should return 'Unknown' for empty values due to default case.
    $this->assertEquals('Unknown', (string) $result);
  }

  /**
   * Tests rendering null referer type.
   *
   * @covers ::render
   */
  public function testRenderNull(): void {
    $values = new ResultRow(['referer_type' => NULL]);
    $this->field->field_alias = 'referer_type';

    $result = $this->field->render($values);
    // Should return 'Unknown' for null values due to default case.
    $this->assertEquals('Unknown', (string) $result);
  }

}
