# Module Settings

Complete guide to configuring the Visitors module settings available in the admin interface.

## Configuration Overview

Access module settings at: **Configuration** → **System** → **Visitors** (`/admin/config/system/visitors`)

## Available Settings

### Visitor Tracking

#### Track Visitors
**Setting**: `disable_tracking`
**Type**: Boolean (radio buttons)
**Default**: `false` (Enabled)
**Options**:
- **Enabled**: Track visitors to the site
- **Disabled**: Turn off all visitor tracking

Controls whether visitor tracking is active globally.

## Tracking Scope

### Page Visibility Settings

#### Pages to Track
**Setting**: `visibility.request_path_pages`
**Type**: Textarea
**Default**: Empty

Specify which pages to track using Drupal path patterns. Enter one path per line.

**Examples**:
```
<front>
/blog
/blog/*
node/*
```

**Path patterns**:
- `*` character is a wildcard
- `<front>` is the front page
- `/blog` for specific pages
- `/blog/*` for all pages under /blog

#### Tracking Mode
**Setting**: `visibility.request_path_mode`
**Type**: Radio buttons
**Options**:
- **All pages except those listed** (0) - Exclude specific pages
- **Only the listed pages** (1) - Include only specific pages

### Role Visibility Settings

#### Role-Based Tracking Mode
**Setting**: `visibility.user_role_mode`
**Type**: Radio buttons
**Options**:
- **Add to the selected roles only** (0)
- **Add to every role except the selected ones** (1)

#### Roles to Track
**Setting**: `visibility.user_role_roles`
**Type**: Checkboxes
**Default**: Empty array

Select which user roles should be tracked (or excluded, depending on the mode above).

### User Account Settings

#### User Account Tracking Mode
**Setting**: `visibility.user_account_mode`
**Type**: Radio buttons
**Options**:
- **Track all users** (0)
- **Track only users with opt-out permission** (1)
- **Do not track users with opt-out permission** (2)

#### Exclude User 1
**Setting**: `visibility.exclude_user1`
**Type**: Checkbox
**Default**: `false`

Exclude the site's first user (admin) from tracking.

## Entity Counter Settings

### Counter Status
**Setting**: `counter.enabled`
**Type**: Checkbox
**Default**: `true`

Enable or disable the entity visit counter functionality.

### Entity Types to Count
**Setting**: `counter.entity_types`
**Type**: Checkboxes
**Default**: `['node']`

Select which entity types should have visit counters:
- **Node**: Content pages
- **User**: User profiles
- **Term**: Taxonomy terms
- **Other entity types**: As available

### Track User ID
**Setting**: `track.userid`
**Type**: Checkbox
**Default**: `true`

Track the user ID of authenticated users in visit data.

## Data Retention

### Log Retention Period
**Setting**: `flush_log_timer`
**Type**: Select dropdown
**Default**: `0` (Never)

Automatically discard visitor logs older than the specified time:

**Options**:
- **Never** (0) - Keep all data
- **1 hour** (3600)
- **3 hours** (10800)
- **6 hours** (21600)
- **9 hours** (32400)
- **12 hours** (43200)
- **1 day** (86400)
- **2 days** (172800)
- **3 days** (259200)
- **1 week** (604800)
- **2 weeks** (1209600)
- **1 month 3 weeks** (4838400)
- **3 months 3 weeks** (9676800)
- **1 year** (31536000)
- **13 months** (34214400)

**Note**: Requires a correctly configured cron maintenance task.

### Bot Data Retention
**Setting**: `bot_retention_log`
**Type**: Select dropdown
**Default**: `0`

How to handle bot and crawler visits:

**Options**:
- **Keep bot data** (0) - Store bot visits normally
- **Discard bot data** (-1) - Don't store bot visits at all
- **Time-based retention** - Same options as log retention above

## Display Settings

### Items Per Page
**Setting**: `items_per_page`
**Type**: Select dropdown
**Default**: `10`

Maximum number of items to display per page in reports:

**Options**: 5, 10, 25, 50, 100, 200, 250, 500, 1000

### Script Type
**Setting**: `script_type`
**Type**: Select
**Default**: `minified`

Which version of the tracking JavaScript to use:
- **Minified**: Compressed version for production
- **Full**: Uncompressed version for development

## Configuration Examples

### Basic Site Configuration
```yaml
# Basic tracking setup
disable_tracking: false
items_per_page: 10
flush_log_timer: 604800  # 1 week retention
bot_retention_log: 0     # Keep bot data
track:
  userid: true
counter:
  enabled: true
  entity_types: ['node']
```

### High-Privacy Configuration
```yaml
# Privacy-focused setup
disable_tracking: false
flush_log_timer: 86400   # 1 day retention only
bot_retention_log: -1    # Discard bot data
visibility:
  user_account_mode: 2   # Don't track users with opt-out
  exclude_user1: true    # Exclude admin user
track:
  userid: false          # Don't track user IDs
```

### Content-Only Tracking
```yaml
# Track only content pages
visibility:
  request_path_mode: 1   # Only listed pages
  request_path_pages: |
    <front>
    node/*
  user_role_mode: 1      # Exclude selected roles
  user_role_roles:
    administrator: administrator
counter:
  enabled: true
  entity_types: ['node']
```

### Development Configuration
```yaml
# Development site setup
disable_tracking: false
flush_log_timer: 3600    # 1 hour retention
bot_retention_log: 0     # Keep bot data for testing
items_per_page: 25       # More items per page
script_type: full        # Unminified JavaScript
```

## Configuration Management

### Export Current Settings
```bash
# Export visitors configuration
drush config:export --destination=/tmp/config

# View current settings
drush config:get visitors.settings
```

### Import Configuration
```bash
# Import configuration
drush config:import --source=/path/to/config

# Set individual values
drush config:set visitors.settings disable_tracking false
drush config:set visitors.settings items_per_page 25
```

### Settings in Code
```php
// Get configuration
$config = \Drupal::config('visitors.settings');
$tracking_disabled = $config->get('disable_tracking');
$items_per_page = $config->get('items_per_page');

// Programmatically update settings
\Drupal::configFactory()->getEditable('visitors.settings')
  ->set('items_per_page', 50)
  ->set('flush_log_timer', 86400)
  ->save();
```

## Validation and Constraints

### Required Settings
- All settings have sensible defaults
- No settings are strictly required
- Tracking can be completely disabled

### Setting Interactions
- **Page visibility**: Mode determines how the pages list is interpreted
- **Role visibility**: Mode determines how the roles list is interpreted
- **User tracking**: Account mode overrides role settings
- **Data retention**: Requires cron to be configured

## Performance Considerations

### High-Traffic Sites
- **Shorter retention periods**: Reduce database size
- **Exclude admin users**: Focus on visitor data
- **Limit entity types**: Track only necessary entities
- **Discard bot data**: Reduce noise in analytics

### Storage Management
- **Regular cleanup**: Configure appropriate retention periods
- **Monitor database growth**: Especially with high traffic
- **Cron scheduling**: Ensure cleanup tasks run regularly

## Troubleshooting Settings

### Common Issues

#### No Data Being Collected
- Check `disable_tracking` is set to `false`
- Verify page visibility settings aren't too restrictive
- Confirm user/role visibility allows tracking
- Check cron is running for data processing

#### Too Much Data
- Reduce `flush_log_timer` for shorter retention
- Set `bot_retention_log` to `-1` to discard bot data
- Limit `counter.entity_types` to essential types only
- Exclude admin users with `exclude_user1`

#### Performance Problems
- Lower `items_per_page` for faster report loading
- Implement shorter data retention periods
- Consider excluding high-traffic user roles
- Monitor database size and optimize accordingly

## Next Steps

- **[Spam Referrer Detection](spam-detection.md)**: Configure referrer filtering
- **[Search Engines](search-engines.md)**: Set up search engine detection



---

**Note**: All settings are stored in the `visitors.settings` configuration object and can be managed through the admin interface or programmatically via Drupal's configuration API.