<?php

namespace Drupal\visitors\Plugin\VisitorsEvent;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\visitors\Attribute\VisitorsEvent;
use Drupal\visitors\Plugin\VisitorsEventBase;
use Drupal\visitors\VisitorsEventPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation for form event.
 */
#[VisitorsEvent(
  id: "form",
  label: "Form",
  weight: 10,
)]
class Form extends VisitorsEventBase implements VisitorsEventPluginInterface, ContainerFactoryPluginInterface {

  /**
   * Constructs a new Form object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The route match service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly RouteMatchInterface $routeMatch,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_route_match'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function process(array $context): ?array {

    $forms = &drupal_static('visitors_form_alter') ?? [];

    if ($forms && count($forms) == 1) {

      $form_id = array_keys($forms)[0];
      $base_form_id = array_values($forms)[0];
      $return['event'] = 'view';
      $return['variables']['plugin_var_3'] = $base_form_id ?? $form_id;
      $return['variables']['plugin_var_4'] = $form_id;

      $form_validate = &drupal_static('visitors_form_validate') ?? [];
      if ($form_validate && count($form_validate) == 1) {
        $return['event'] = 'validate';
      }

      $form_submit = &drupal_static('visitors_form_submit') ?? [];
      if ($form_submit && count($form_submit) == 1) {
        $return['event'] = 'submit';
      }

      return $return;
    }

    return NULL;
  }

  /**
   * A validation callback for forms.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function validate(array &$form, FormStateInterface $form_state) {
    $static = &drupal_static('visitors_form_validate');

    $form_object = $form_state->getFormObject();
    if (!$form_object) {
      return;
    }

    $form_id = $form_object->getFormId();
    $base_form_id = $form_id;
    if (method_exists($form_object, 'getBaseFormId')) {
      $base_form_id = $form_object->getBaseFormId();
    }

    $static[$form_id] = $base_form_id;
  }

  /**
   * A submit callback for forms.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function submit(array &$form, FormStateInterface $form_state) {
    $static = &drupal_static('visitors_form_submit');

    $form_object = $form_state->getFormObject();
    if (!$form_object) {
      return;
    }

    $form_id = $form_object->getFormId();
    $base_form_id = $form_id;
    if (method_exists($form_object, 'getBaseFormId')) {
      $base_form_id = $form_object->getBaseFormId();
    }

    $static[$form_id] = $base_form_id;
  }

}
