<?php

namespace Drupal\visitors\Plugin\views\area;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\EventSubscriber\AjaxResponseSubscriber;
use Drupal\Core\EventSubscriber\MainContentViewSubscriber;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\views\Attribute\ViewsArea;
use Drupal\views\Plugin\views\area\DisplayLink;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Views area display_link handler.
 *
 * @ingroup views_area_handlers
 */
#[ViewsArea("visitors_display_link")]
class VisitorsDisplayLink extends DisplayLink {

  /**
   * The view settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $viewSettings;

  /**
   * Constructs a new VisitorsDisplayLink object.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Config\ImmutableConfig $view_settings
   *   The view settings.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ImmutableConfig $view_settings) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->viewSettings = $view_settings;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory')->get('views.settings')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['display_links'] = ['default' => []];
    $options['render_format'] = ['default' => 'link'];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    $allowed_displays = [];
    $displays = $this->view->storage->get('display');
    foreach ($displays as $display_id => $display) {
      if ($this->isPathBasedDisplay($display_id)) {
        unset($displays[$display_id]);
        continue;
      }
      $allowed_displays[$display_id] = $display['display_title'];
    }

    $form['description'] = [
      [
        '#markup' => $this->t('To make sure the results are the same when switching to the other display, it is recommended to make sure the display:'),
      ],
      [
        '#theme' => 'item_list',
        '#items' => [
          $this->t('Has a path.'),
          $this->t('Has the same filter criteria.'),
          $this->t('Has the same sort criteria.'),
          $this->t('Has the same contextual filters.'),
        ],
      ],
    ];

    if (!$allowed_displays) {
      $form['empty_message'] = [
        '#markup' => '<p><em>' . $this->t('There are no path-based displays available.') . '</em></p>',
      ];
    }
    else {
      // Render format selection.
      $form['render_format'] = [
        '#title' => $this->t('Render format'),
        '#type' => 'select',
        '#options' => [
          'link' => $this->t('Single link (if only one display)'),
          'radio' => $this->t('Radio buttons'),
          'checkboxes' => $this->t('Checkboxes'),
          'list' => $this->t('Unordered list of links'),
        ],
        '#default_value' => (string) ($this->options['render_format'] ?? 'link'),
        '#description' => $this->t('Choose how to render the display links.'),
      ];

      // Display links configuration.
      $form['display_links'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Display Links'),
        '#description' => $this->t('Configure multiple display links. Each link can have its own label.'),
        '#tree' => TRUE,
      ];

      $display_links = $this->options['display_links'] ?? [];
      if (empty($display_links)) {
        // Initialize with one empty link if none exist.
        $display_links = [['display_id' => '', 'label' => '']];
      }
      else {
        // Always add an empty link at the end for adding new ones.
        $display_links[] = ['display_id' => '', 'label' => ''];
      }

      $total_links = count($display_links);
      foreach ($display_links as $index => $link) {
        // Ensure we have a valid array structure.
        if (!is_array($link)) {
          continue;
        }

        $is_empty = empty((string) ($link['display_id'] ?? '')) && empty((string) ($link['label'] ?? ''));
        $is_last = ($index === $total_links - 1);

        $form['display_links'][$index] = [
          '#type' => 'fieldset',
          '#title' => $is_empty ? $this->t('Add new link') : $this->t('Link @index', ['@index' => (int) $index + 1]),
          '#collapsible' => TRUE,
          '#collapsed' => $is_empty ? FALSE : ($index > 0),
        ];

        $form['display_links'][$index]['display_id'] = [
          '#title' => $this->t('Display'),
          '#type' => 'select',
          '#options' => ['' => $this->t('- Select -')] + $allowed_displays,
          '#default_value' => (string) ($link['display_id'] ?? ''),
          '#required' => !$is_empty,
        ];

        $form['display_links'][$index]['label'] = [
          '#title' => $this->t('Label'),
          '#description' => $this->t('The text of the link.'),
          '#type' => 'textfield',
          '#default_value' => (string) ($link['label'] ?? ''),
          '#required' => !$is_empty,
        ];

        // Only show remove button for non-empty links that are not the last
        // one.
        if (!$is_empty && !$is_last) {
          $form['display_links'][$index]['remove'] = [
            '#type' => 'submit',
            '#value' => $this->t('Remove'),
            '#name' => 'remove_link_' . $index,
            '#submit' => [[$this, 'removeDisplayLink']],
          ];
        }
      }

    }
  }

  /**
   * AJAX callback to remove a display link.
   */
  public function removeDisplayLink(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $index = (int) str_replace('remove_link_', '', $triggering_element['#name']);

    $display_links = $form_state->getValue('display_links', []);
    unset($display_links[$index]);
    // Re-index array.
    $display_links = array_values($display_links);
    $form_state->setValue('display_links', $display_links);
    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function validate() {
    $errors = [];

    // Do not add errors for the default display if it is not displayed in the
    // UI.
    if ($this->displayHandler->isDefaultDisplay() && !$this->viewSettings->get('ui.show.default_display')) {
      return $errors;
    }

    // Validate new display links configuration.
    $display_links = $this->options['display_links'] ?? [];
    if (!empty($display_links)) {
      foreach ($display_links as $index => $link) {
        // Ensure we have valid array structure and convert values to strings.
        if (!is_array($link) || empty($link['display_id']) || empty($link['label'])) {
          // Skip empty links.
          continue;
        }

        $linked_display_id = (string) $link['display_id'];
        $link_label = (string) $link['label'];

        // Check if the linked display hasn't been removed.
        if (!$this->view->displayHandlers->get($linked_display_id)) {
          $errors[] = $this->t('%current_display: Link @index in the %area area points to the %linked_display display which no longer exists.', [
            '%current_display' => $this->displayHandler->display['display_title'],
            '@index' => $index + 1,
            '%area' => $this->areaType,
            '%linked_display' => $linked_display_id,
          ]);
          continue;
        }

        // Check if the linked display is a path-based display.
        if ($this->isPathBasedDisplay($linked_display_id)) {
          $errors[] = $this->t('%current_display: Link @index in the %area area points to the %linked_display display which does not have a path.', [
            '%current_display' => $this->displayHandler->display['display_title'],
            '@index' => $index + 1,
            '%area' => $this->areaType,
            '%linked_display' => $this->view->displayHandlers->get($linked_display_id)->display['display_title'],
          ]);
          continue;
        }

        // Check if options of the linked display are equal to the options of
        // the current display. We "only" show a warning here, because even
        // though we recommend keeping the display options equal, we do not want
        // to enforce this.
        $unequal_options = [
          'filters' => $this->t('Filter criteria'),
          'arguments' => $this->t('Contextual filters'),
        ];
        foreach (array_keys($unequal_options) as $option) {
          if ($this->hasEqualOptions($linked_display_id, $option)) {
            unset($unequal_options[$option]);
          }
        }

        if ($unequal_options) {
          $warning = $this->t('%current_display: Link @index in the %area area points to the %linked_display display which uses different settings than the %current_display display for: %unequal_options. To make sure users see the exact same result when clicking the link, please check that the settings are the same.', [
            '%current_display' => $this->displayHandler->display['display_title'],
            '@index' => $index + 1,
            '%area' => $this->areaType,
            '%linked_display' => $this->view->displayHandlers->get($linked_display_id)->display['display_title'],
            '%unequal_options' => implode(', ', $unequal_options),
          ]);
          $this->messenger()->addWarning($warning);
        }
      }

      // If we have display links configured, return any errors found.
      if (!empty($errors)) {
        return $errors;
      }
    }

    // If no display links are configured at all, show an error.
    if (empty($display_links)) {
      $errors[] = $this->t('%current_display: The link in the %area area has no configured display.', [
        '%current_display' => $this->displayHandler->display['display_title'],
        '%area' => $this->areaType,
      ]);
    }

    return $errors;
  }

  /**
   * {@inheritdoc}
   */
  public function render($empty = FALSE) {
    if ($empty && empty($this->options['empty'])) {
      return [];
    }

    // Get display links configuration.
    $display_links = $this->options['display_links'] ?? [];
    $render_format = $this->options['render_format'] ?? 'link';

    // Filter out empty links.
    $display_links = array_filter($display_links, function ($link) {
      return is_array($link) && !empty($link['display_id']) && !empty($link['label']);
    });

    if (empty($display_links)) {
      return [];
    }

    // Filter out path-based displays.
    $valid_links = [];
    foreach ($display_links as $link) {
      if (!$this->isPathBasedDisplay($link['display_id'])) {
        $valid_links[] = $link;
      }
    }

    if (empty($valid_links)) {
      return [];
    }

    // If only one link and format is 'link', render as single link.
    if (count($valid_links) === 1 && $render_format === 'link') {
      return $this->renderSingleLink($valid_links[0]);
    }

    // Render based on format.
    switch ($render_format) {
      case 'radio':
        return $this->renderRadioButtons($valid_links);

      case 'checkboxes':
        return $this->renderCheckboxes($valid_links);

      case 'list':
        return $this->renderUnorderedList($valid_links);

      case 'link':
      default:
        return $this->renderUnorderedList($valid_links);
    }
  }

  /**
   * Renders a single display link.
   *
   * @param array $link
   *   The link configuration.
   *
   * @return array
   *   The render array for a single link.
   */
  protected function renderSingleLink(array $link): array {
    $label = (string) $link['label'];
    $display_id = (string) $link['display_id'];

    // Get query parameters from the exposed input and pager.
    $query = $this->view->getExposedInput();
    if ($current_page = $this->view->getCurrentPage()) {
      $query['page'] = $current_page;
    }

    // @todo Remove this parsing once these are removed from the request in
    //   https://www.drupal.org/node/2504709.
    foreach ([
      'view_name',
      'view_display_id',
      'view_args',
      'view_path',
      'view_dom_id',
      'pager_element',
      'view_base_path',
      AjaxResponseSubscriber::AJAX_REQUEST_PARAMETER,
      FormBuilderInterface::AJAX_FORM_REQUEST,
      MainContentViewSubscriber::WRAPPER_FORMAT,
    ] as $key) {
      unset($query[$key]);
    }

    $storage_id = $this->view->storage->id();

    // Set default classes.
    $classes = [
      'views-display-link',
      'views-display-link-' . $display_id,
    ];
    if ($display_id === $this->view->current_display) {
      $classes[] = 'is-active';
    }
    $classes[] = 'use-ajax';

    $path = 'internal:/admin/visitors/_report/' . $storage_id . '/' . $display_id;
    $query_class = '.view-id-' . $storage_id . '.view-display-id-' . $this->view->current_display;

    $query_params = ['class' => $query_class];

    return [
      '#type' => 'link',
      '#title' => $label,
      '#url' => Url::fromUri($path, [
        'query' => $query_params,
      ]),
      '#options' => [
        'attributes' => ['class' => $classes],
      ],
    ];
  }

  /**
   * Renders display links as radio buttons.
   *
   * @param array $links
   *   Array of link configurations.
   *
   * @return array
   *   The render array for radio buttons.
   */
  protected function renderRadioButtons(array $links): array {
    $options = [];
    $default_value = NULL;

    foreach ($links as $link) {
      $display_id = (string) $link['display_id'];
      $label = (string) $link['label'];

      $options[$display_id] = $label;

      if ($display_id === $this->view->current_display) {
        $default_value = $display_id;
      }
    }

    return [
      '#type' => 'radios',
      '#title' => $this->t('Display Options'),
      '#options' => $options,
      '#default_value' => $default_value,
      '#attributes' => [
        'class' => ['visitors-display-radios'],
        'data-ajax' => 'true',
      ],
      '#ajax' => [
        'callback' => [$this, 'ajaxDisplayChange'],
        'wrapper' => 'visitors-display-content',
      ],
    ];
  }

  /**
   * Renders display links as checkboxes.
   *
   * @param array $links
   *   Array of link configurations.
   *
   * @return array
   *   The render array for checkboxes.
   */
  protected function renderCheckboxes(array $links): array {
    $options = [];
    $default_value = [];

    foreach ($links as $link) {
      $display_id = (string) $link['display_id'];
      $label = (string) $link['label'];

      $options[$display_id] = $label;

      if ($display_id === $this->view->current_display) {
        $default_value[] = $display_id;
      }
    }

    return [
      '#type' => 'checkboxes',
      '#title' => $this->t('Display Options'),
      '#options' => $options,
      '#default_value' => $default_value,
      '#attributes' => [
        'class' => ['visitors-display-checkboxes'],
        'data-ajax' => 'true',
      ],
      '#ajax' => [
        'callback' => [$this, 'ajaxDisplayChange'],
        'wrapper' => 'visitors-display-content',
      ],
    ];
  }

  /**
   * Renders display links as an unordered list.
   *
   * @param array $links
   *   Array of link configurations.
   *
   * @return array
   *   The render array for an unordered list.
   */
  protected function renderUnorderedList(array $links): array {
    $items = [];

    foreach ($links as $link) {
      $items[] = $this->renderSingleLink($link);
    }

    return [
      '#theme' => 'item_list',
      '#items' => $items,
      '#attributes' => [
        'class' => ['visitors-display-links'],
      ],
    ];
  }

  /**
   * AJAX callback for display change.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The AJAX response.
   */
  public function ajaxDisplayChange(array &$form, FormStateInterface $form_state) {
    // This would need to be implemented based on specific requirements
    // for handling display changes via AJAX.
    return [];
  }

}
