<?php

declare(strict_types=1);

namespace Drupal\Tests\visitors\Kernel\Plugin\views\field;

use Drupal\Core\Database\Connection;
use Drupal\KernelTests\KernelTestBase;
use Drupal\visitors\Plugin\views\field\Bounce;

/**
 * Kernel tests for Bounce views field plugin.
 *
 * @group visitors
 * @coversDefaultClass \Drupal\visitors\Plugin\views\field\Bounce
 */
class BounceTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'views',
    'visitors',
  ];

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The field plugin instance.
   *
   * @var \Drupal\visitors\Plugin\views\field\Bounce
   */
  protected Bounce $field;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installSchema('system', ['sequences']);
    $this->installSchema('visitors', ['visitors_visit', 'visitors_event']);
    $this->installConfig(['user', 'system', 'views']);

    $this->database = $this->container->get('database');

    // Insert test data.
    $this->insertTestData();

    // Create the field plugin instance.
    $this->field = $this->createFieldPlugin();
  }

  /**
   * Inserts test data into the visitors_visit table.
   */
  protected function insertTestData(): void {
    $current_time = time();
    $test_data = [
      // Bounce visit (1 page view).
      [
        'id' => 1,
        'visitor_id' => 'visitor1',
        'uid' => 0,
        'total_page_views' => 1,
        'entry_time' => $current_time,
        'exit_time' => $current_time + 30,
        'location_ip' => '127.0.0.1',
        'config_id' => 'config1',
      ],
      // Non-bounce visit (multiple page views).
      [
        'id' => 2,
        'visitor_id' => 'visitor2',
        'uid' => 0,
        'total_page_views' => 5,
        'entry_time' => $current_time - 3600,
        'exit_time' => $current_time - 3300,
        'location_ip' => '127.0.0.2',
        'config_id' => 'config2',
      ],
      // Another bounce visit.
      [
        'id' => 3,
        'visitor_id' => 'visitor3',
        'uid' => 0,
        'total_page_views' => 1,
        'entry_time' => $current_time - 7200,
        'exit_time' => $current_time - 7100,
        'location_ip' => '127.0.0.3',
        'config_id' => 'config3',
      ],
      // Another non-bounce visit.
      [
        'id' => 4,
        'visitor_id' => 'visitor4',
        'uid' => 0,
        'total_page_views' => 3,
        'entry_time' => $current_time - 14400,
        'exit_time' => $current_time - 14100,
        'location_ip' => '127.0.0.4',
        'config_id' => 'config4',
      ],
    ];

    foreach ($test_data as $data) {
      $this->database->insert('visitors_visit')
        ->fields($data)
        ->execute();
    }
  }

  /**
   * Creates a field plugin instance for testing.
   */
  protected function createFieldPlugin(): Bounce {
    $configuration = [
      'table' => 'visitors_visit',
      'field' => 'total_page_views',
    ];
    $plugin_id = 'visitors_bounce';
    $plugin_definition = [];
    $field = Bounce::create($this->container, $configuration, $plugin_id, $plugin_definition);

    // Set up the field properties.
    $field->tableAlias = 'vv';
    $field->realField = 'total_page_views';

    return $field;
  }

  /**
   * Tests bounce rate calculation through SQL query.
   *
   * @covers ::query
   */
  public function testBounceRateCalculation(): void {
    // Query to test the bounce calculation logic.
    $query = $this->database->select('visitors_visit', 'vv');
    $query->addExpression('CASE WHEN vv.total_page_views = 1 THEN 1 ELSE 0 END', 'bounce_value');
    $query->addField('vv', 'total_page_views', 'original_page_views');
    $query->orderBy('vv.id');
    $results = $query->execute()->fetchAll();

    $this->assertCount(4, $results);

    // Test each result matches expected bounce calculation.
    $expected_results = [
    // Id 1.
      ['bounce_value' => '1', 'original_page_views' => '1'],
    // Id 2.
      ['bounce_value' => '0', 'original_page_views' => '5'],
    // Id 3.
      ['bounce_value' => '1', 'original_page_views' => '1'],
    // Id 4.
      ['bounce_value' => '0', 'original_page_views' => '3'],
    ];

    foreach ($expected_results as $index => $expected) {
      $this->assertEquals($expected['bounce_value'], $results[$index]->bounce_value);
      $this->assertEquals($expected['original_page_views'], $results[$index]->original_page_views);
    }
  }

  /**
   * Tests Views plugin manager integration.
   *
   * @covers ::create
   */
  public function testViewsPluginManagerIntegration(): void {
    // Verify the field plugin is available.
    $field_plugins = \Drupal::service('plugin.manager.views.field')->getDefinitions();
    $this->assertArrayHasKey('visitors_bounce', $field_plugins);

    // Verify plugin definition.
    $plugin_definition = $field_plugins['visitors_bounce'];
    $this->assertEquals('Drupal\visitors\Plugin\views\field\Bounce', $plugin_definition['class']);
    $this->assertEquals('visitors_bounce', $plugin_definition['id']);

    // Test creating plugin through plugin manager.
    $plugin_manager = \Drupal::service('plugin.manager.views.field');
    $plugin = $plugin_manager->createInstance('visitors_bounce');
    $this->assertInstanceOf(Bounce::class, $plugin);
  }

  /**
   * Tests database integration by querying actual bounce data.
   *
   * @covers ::query
   */
  public function testDatabaseIntegration(): void {
    // Count bounces using direct SQL (total_page_views = 1).
    $bounce_count = $this->database->select('visitors_visit', 'vv')
      ->condition('vv.total_page_views', 1)
      ->countQuery()
      ->execute()
      ->fetchField();

    $this->assertEquals(2, $bounce_count);

    // Count non-bounces using direct SQL (total_page_views > 1).
    $non_bounce_count = $this->database->select('visitors_visit', 'vv')
      ->condition('vv.total_page_views', 1, '>')
      ->countQuery()
      ->execute()
      ->fetchField();

    $this->assertEquals(2, $non_bounce_count);

    // Test the CASE WHEN logic directly.
    $query = $this->database->select('visitors_visit', 'vv');
    $query->addExpression('SUM(CASE WHEN vv.total_page_views = 1 THEN 1 ELSE 0 END)', 'bounce_sum');
    $query->addExpression('SUM(CASE WHEN vv.total_page_views > 1 THEN 1 ELSE 0 END)', 'non_bounce_sum');
    $result = $query->execute()->fetch();

    $this->assertEquals(2, $result->bounce_sum);
    $this->assertEquals(2, $result->non_bounce_sum);
  }

}
