<?php

declare(strict_types=1);

namespace Drupal\Tests\visitors\Unit\Plugin\views\area;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Path\PathValidatorInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\views\DisplayPluginCollection;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\ViewEntityInterface;
use Drupal\views\ViewExecutable;
use Drupal\visitors\Plugin\views\area\VisitorsDisplayLink;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Tests the VisitorsDisplayLink plugin.
 *
 * @coversDefaultClass \Drupal\visitors\Plugin\views\area\VisitorsDisplayLink
 * @group visitors
 */
final class VisitorsDisplayLinkTest extends UnitTestCase {

  /**
   * The plugin under test.
   *
   * @var \Drupal\visitors\Plugin\views\area\VisitorsDisplayLink
   */
  protected VisitorsDisplayLink $plugin;

  /**
   * The view executable mock.
   *
   * @var \Drupal\views\ViewExecutable|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $view;

  /**
   * The display handler mock.
   *
   * @var \Drupal\views\Plugin\views\display\DisplayPluginBase|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $displayHandler;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $container->set('string_translation', $this->getStringTranslationStub());

    // Mock path validator service.
    $path_validator = $this->createMock(PathValidatorInterface::class);
    $container->set('path.validator', $path_validator);

    \Drupal::setContainer($container);

    $this->view = $this->createMock(ViewExecutable::class);
    $this->displayHandler = $this->createMock(DisplayPluginBase::class);

    $configuration = [
      'display_id' => 'test_display',
      'label' => 'Test Link',
    ];

    $plugin_definition = [
      'id' => 'visitors_display_link',
      'provider' => 'visitors',
    ];

    $view_settings = $this->createMock(ImmutableConfig::class);
    $view_settings->method('get')
      ->with('ui.show.default_display')
      ->willReturn(TRUE);

    $this->plugin = new VisitorsDisplayLink(
      $configuration,
      'visitors_display_link',
      $plugin_definition,
      $view_settings
    );

    $this->plugin->view = $this->view;
    $this->plugin->displayHandler = $this->displayHandler;

    // Mock the string translation service.
    $translation = $this->createMock(TranslationInterface::class);
    $translation->method('translate')
      ->willReturnCallback(function ($string) {
        return $string;
      });
    $this->plugin->setStringTranslation($translation);
  }

  /**
   * Tests the defineOptions method.
   *
   * @covers ::defineOptions
   */
  public function testDefineOptions(): void {
    // Use reflection to access the protected method.
    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('defineOptions');
    $method->setAccessible(TRUE);

    $options = $method->invoke($this->plugin);
    $this->assertArrayHasKey('display_id', $options);
    $this->assertEquals('', $options['display_id']['default']);
    $this->assertArrayHasKey('label', $options);
    $this->assertEquals('', $options['label']['default']);
  }

  /**
   * Tests the buildOptionsForm method.
   *
   * @covers ::buildOptionsForm
   */
  public function skippedTestBuildOptionsForm(): void {
    $form = [];
    $form_state = $this->createMock(FormStateInterface::class);

    // Mock the view storage and displays.
    $view_storage = $this->createMock(ViewEntityInterface::class);
    $this->view->storage = $view_storage;

    $displays = [
      'default' => [
        'display_title' => 'Default',
        'display_plugin' => 'default',
      ],
      'page_1' => [
        'display_title' => 'Page 1',
        'display_plugin' => 'page',
      ],
    ];

    $view_storage->method('get')
      ->with('display')
      ->willReturn($displays);

    // Mock display handlers using DisplayPluginCollection.
    $default_display = $this->createMock(DisplayPluginBase::class);
    $page_display = $this->createMock(DisplayPluginBase::class);

    $display_collection = $this->getMockBuilder(DisplayPluginCollection::class)
      ->disableOriginalConstructor()
      ->getMock();

    $this->view->displayHandlers = $display_collection;

    // Set protected properties using the helper method.
    $this->setProtectedProperty('view', $this->view);
    $this->setProtectedProperty('displayHandler', $this->displayHandler);

    // Call the method.
    $this->plugin->buildOptionsForm($form, $form_state);

  }

  /**
   * Sets a protected property on the plugin for testing.
   *
   * @param string $property
   *   The property name.
   * @param mixed $value
   *   The property value.
   */
  protected function setProtectedProperty(string $property, $value): void {
    $reflection = new \ReflectionClass($this->plugin);
    $property_reflection = $reflection->getProperty($property);
    $property_reflection->setAccessible(TRUE);
    $property_reflection->setValue($this->plugin, $value);
  }

}
