<?php

declare(strict_types=1);

namespace Drupal\Tests\visitors\Unit\Plugin\views\field;

use Drupal\Tests\UnitTestCase;
use Drupal\views\Plugin\views\query\Sql;
use Drupal\visitors\Plugin\views\field\Bounce;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Unit tests for Bounce views field plugin.
 *
 * @group visitors
 * @coversDefaultClass \Drupal\visitors\Plugin\views\field\Bounce
 */
class BounceTest extends UnitTestCase {

  /**
   * The field plugin instance.
   *
   * @var \Drupal\visitors\Plugin\views\field\Bounce
   */
  protected Bounce $field;

  /**
   * Mock query plugin.
   *
   * @var \Drupal\views\Plugin\views\query\Sql|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $query;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    \Drupal::setContainer($container);

    // Create the field plugin.
    $configuration = [
      'table' => 'visitors_visit',
      'field' => 'bounce',
    ];
    $plugin_id = 'visitors_bounce';
    $plugin_definition = [];
    $this->field = Bounce::create($container, $configuration, $plugin_id, $plugin_definition);

    // Mock the query plugin.
    $this->query = $this->createMock(Sql::class);
    $this->field->query = $this->query;

    // Set up field properties.
    $this->field->tableAlias = 'vv';
    $this->field->realField = 'bounce';
  }

  /**
   * Tests field plugin instantiation.
   *
   * @covers ::create
   */
  public function testCreate(): void {
    $configuration = [];
    $plugin_id = 'visitors_bounce';
    $plugin_definition = [];

    $container = new ContainerBuilder();
    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $field = Bounce::create($container, $configuration, $plugin_id, $plugin_definition);
    $this->assertInstanceOf(Bounce::class, $field);
  }

  /**
   * Tests the query method to ensure proper SQL expression is added.
   *
   * @covers ::query
   */
  public function testQuery(): void {
    // Mock addField to capture the SQL expression.
    $this->query->expects($this->once())
      ->method('addField')
      ->with(
        NULL,
        'CASE WHEN vv.bounce = 1 THEN 1 ELSE 0 END',
        'vv_bounce'
      )
      ->willReturn('vv_bounce');

    // Call the query method.
    $this->field->query();

    // Assert field_alias is set.
    $this->assertEquals('vv_bounce', $this->field->field_alias);
  }

  /**
   * Tests query method with different table alias and field name.
   *
   * @covers ::query
   */
  public function testQueryWithCustomAliasAndField(): void {
    // Set up field properties with custom values.
    $this->field->tableAlias = 'custom_table';
    $this->field->realField = 'is_bounce';

    // Mock addField with custom values.
    $this->query->expects($this->once())
      ->method('addField')
      ->with(
        NULL,
        'CASE WHEN custom_table.is_bounce = 1 THEN 1 ELSE 0 END',
        'custom_table_is_bounce'
      )
      ->willReturn('custom_table_is_bounce');

    // Call the query method.
    $this->field->query();

    // Assert field_alias is set correctly.
    $this->assertEquals('custom_table_is_bounce', $this->field->field_alias);
  }

  /**
   * Tests that the plugin extends the correct base class.
   */
  public function testPluginInheritance(): void {
    $this->assertInstanceOf('Drupal\views\Plugin\views\field\FieldPluginBase', $this->field);
  }

}
