<?php

namespace Drupal\vnts\Plugin\views\filter;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\filter\FilterPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;

/**
 * Provides a filter with a dropdown/checklist/autocomplete of node titles.
 *
 * @ViewsFilter("node_title_select")
 */
class NodeTitleSelect extends FilterPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a NodeTitleSelect object.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defineOptions() {
    $options = parent::defineOptions();
    $options['show_unpublished'] = ['default' => FALSE];
    $options['excluded_nids'] = ['default' => []];
    $options['include_new_nodes'] = ['default' => TRUE];
    $options['widget'] = ['default' => 'select'];
    $options['frozen_nids'] = ['default' => []];
    return $options;
  }

  /**
   * Build the options form in Views UI.
   *
   * @inheritdoc
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    // Calling parent::buildOptionsForm() is crucial as it adds the standard
    // filter options like "Expose this filter". We will then reorder things.
    parent::buildOptionsForm($form, $form_state);

    // Get the standard exposed and required form elements to be re-ordered.
    $exposed_options = $form['expose'] ?? [];
    $required_option = $form['is_exposed_required'] ?? [];

    // Remove the default elements so we can re-add them in a new order.
    unset($form['expose'], $form['is_exposed_required']);

    // Add the exposed filter options first, but without the description.
    if (!empty($exposed_options)) {
      $exposed_options['#description'] = '';
      $form['expose'] = $exposed_options;
    }

    // Add the "required" checkbox back, if it was present, but without the description.
    if (!empty($required_option)) {
      $required_option['#description'] = '';
      $form['is_exposed_required'] = $required_option;
    }

    // Now, add the custom elements in the desired order.
    $form['widget'] = [
      '#type' => 'select',
      '#title' => $this->t('Exposed widget type'),
      '#options' => [
        'select' => $this->t('Select list (single)'),
        'autocomplete' => $this->t('Autocomplete (entity_autocomplete)'),
        'checkboxes' => $this->t('Checkboxes (multi-select)'),
      ],
      '#default_value' => $this->options['widget'] ?? 'select',
      '#weight' => -10, // Make this appear on top.
    ];

    $form['include_new_nodes'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Automatically include newly created nodes'),
      '#default_value' => $this->options['include_new_nodes'] ?? TRUE,
      '#weight' => -9,
    ];

    // Detect bundles from the current view.
    $detected_bundles = $this->detectBundlesFromView();
    $detected_labels = [];
    foreach ($detected_bundles as $b) {
      $t = $this->entityTypeManager->getStorage('node_type')->load($b);
      $detected_labels[] = $t ? $t->label() : $b;
    }

    $form['detected_bundles_info'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<strong>Note:</strong> This filter will display nodes from the following content types already configured in this view: @list. If this list is empty, all nodes will be displayed.', ['@list' => implode(', ', $detected_labels)]),
      '#weight' => -8,
    ];

    $form['show_unpublished'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include unpublished nodes'),
      '#default_value' => $this->options['show_unpublished'] ?? FALSE,
      '#weight' => -7,
    ];

    // Build the list of node options to allow admin to exclude specific nodes.
    $node_options = $this->buildAdminNodeOptions($detected_bundles, $this->options['show_unpublished'] ?? FALSE);

    // Excluded nodes from the list.
    $form['excluded_nids'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Exclude nodes from the exposed list'),
      '#description' => $this->t('Check nodes you do NOT want to appear in the exposed filter dropdown/list.'),
      '#options' => $node_options,
      '#default_value' => $this->options['excluded_nids'] ?? [],
      '#weight' => -6,
    ];

    // If the admin wants a frozen list (include_new_nodes FALSE), show currently frozen nids.
    if (empty($this->options['include_new_nodes'])) {
      $form['frozen_info'] = [
        '#type' => 'markup',
        '#markup' => $this->t('This filter is configured to use a frozen list of nodes (no new nodes will be auto-included).'),
        '#weight' => -5,
      ];
    }

    // The rest of the form elements (like the "Exposed identifier") will
    // be added by the parent::buildOptionsForm() and will appear after
    // our custom elements due to the negative weight values.
  }

  /**
   * Submit handler for the Views UI options form. (Views will call this.)
   *
   * @inheritdoc
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state) {
    parent::submitOptionsForm($form, $form_state);
    $values = $form_state->getValues();

    // No need to save the 'bundles' option anymore.
    unset($this->options['bundles']);

    // Save unpublished flag.
    $this->options['show_unpublished'] = !empty($values['show_unpublished']);

    // Save widget type.
    $this->options['widget'] = $values['widget'] ?? 'select';

    // Save include_new_nodes flag.
    $this->options['include_new_nodes'] = !empty($values['include_new_nodes']);

    // Save excluded nodes (normalize to integer keys).
    $excluded = $values['excluded_nids'] ?? [];
    $excluded_nids = array_keys(array_filter((array) $excluded));
    $this->options['excluded_nids'] = array_map('intval', $excluded_nids);

    // If include_new_nodes is FALSE, freeze the current node list to the current
    // node IDs (minus excluded); this creates a stable list.
    if (empty($this->options['include_new_nodes'])) {
      $detected_bundles = $this->detectBundlesFromView();
      $current_nids = array_keys($this->buildAdminNodeOptions($detected_bundles, $this->options['show_unpublished']));
      // Remove excluded from frozen list.
      $frozen = array_diff($current_nids, $this->options['excluded_nids']);
      $this->options['frozen_nids'] = array_values(array_map('intval', $frozen));
    }
    else {
      // Not frozen.
      $this->options['frozen_nids'] = [];
    }
  }

  /**
   * Helper: detect bundles (content types) from other filters configured in the view.
   *
   * Looks for a filter with field 'type' and extracts its value(s) (selected bundles).
   * Returns an array of bundle machine names.
   */
  protected function detectBundlesFromView(): array {
    $bundles = [];
    if (empty($this->view) || empty($this->view->display_handler)) {
      return $bundles;
    }

    $display_handler = $this->view->display_handler;
    $filters = $display_handler->getOption('filters') ?? [];

    foreach ($filters as $filter_info) {
      if (($filter_info['table'] ?? '') === 'node_field_data' && ($filter_info['field'] ?? '') === 'type') {
        $value = $filter_info['value'] ?? [];
        if (is_array($value)) {
          $bundles = array_merge($bundles, array_values(array_filter($value)));
        }
      }
    }
    // Unique and return.
    return array_values(array_unique(array_filter($bundles)));
  }

  /**
   * Build admin node options for the configuration UI (nid => title).
   *
   * @param array $detected_bundles
   * Bundles detected from the view (may be empty).
   * @param bool $include_unpublished
   * Whether to include unpublished nodes.
   *
   * @return array
   * Associative array [nid => title].
   */
  protected function buildAdminNodeOptions(array $detected_bundles = [], bool $include_unpublished = FALSE): array {
    $storage = $this->entityTypeManager->getStorage('node');
    $query = $storage->getQuery()->accessCheck(TRUE);

    if (!empty($detected_bundles)) {
      $query->condition('type', $detected_bundles, 'IN');
    }

    if (!$include_unpublished) {
      $query->condition('status', 1);
    }

    $query->sort('title', 'ASC');
    $nids = $query->execute();

    $options = [];
    if (!empty($nids)) {
      $nodes = $storage->loadMultiple($nids);
      foreach ($nodes as $nid => $node) {
        $options[$nid] = $node->label();
      }
    }
    return $options;
  }

  /**
   * Returns the options shown to end users on the exposed/filter widget.
   *
   * Honors include_new_nodes, excluded_nids and frozen_nids.
   * Uses per-request static caching to avoid repeated queries.
   *
   * @return array
   *   [nid => title]
   */
  protected function getOptions(): array {
    // Static cache to avoid multiple DB queries in the same request.
    static $cache = [];

    // Build a unique cache key based on filter settings.
    $cache_key = md5(serialize([
      'include_new_nodes' => $this->options['include_new_nodes'] ?? TRUE,
      'frozen_nids'       => $this->options['frozen_nids'] ?? [],
      'excluded_nids'     => $this->options['excluded_nids'] ?? [],
      'show_unpublished'  => $this->options['show_unpublished'] ?? FALSE,
      'bundles'           => $this->detectBundlesFromView(),
    ]));

    if (isset($cache[$cache_key])) {
      return $cache[$cache_key];
    }

    // Only use frozen list if include_new_nodes is FALSE.
    if (empty($this->options['include_new_nodes']) && !empty($this->options['frozen_nids'])) {
      $nids = array_diff($this->options['frozen_nids'], $this->options['excluded_nids'] ?? []);
      if (empty($nids)) {
        return $cache[$cache_key] = [];
      }
      $nodes = $this->entityTypeManager->getStorage('node')->loadMultiple($nids);
      $options = [];
      foreach ($nids as $nid) {
        if (!empty($nodes[$nid])) {
          $options[$nid] = $nodes[$nid]->label();
        }
      }
      return $cache[$cache_key] = $options;
    }

    // Otherwise, build dynamic list (always fresh).
    $detected_bundles = $this->detectBundlesFromView();
    $node_options = $this->buildAdminNodeOptions($detected_bundles, $this->options['show_unpublished'] ?? FALSE);

    // Remove excluded nids.
    $excluded = $this->options['excluded_nids'] ?? [];
    if (!empty($excluded)) {
      foreach ($excluded as $eid) {
        unset($node_options[$eid]);
      }
    }

    return $cache[$cache_key] = $node_options;
  }


  /**
   * Build the exposed form element (select, autocomplete or checkboxes).
   *
   * @inheritdoc
   */
  public function buildExposedForm(&$form, FormStateInterface $form_state) {
    parent::buildExposedForm($form, $form_state);

    $options = $this->getOptions();
    $identifier = $this->options['expose']['identifier'] ?? ('node_title_select_' . $this->definition['id']);
    $widget = $this->options['widget'] ?? 'select';

    switch ($widget) {
      case 'autocomplete':
        $form[$identifier] = [
          '#type' => 'entity_autocomplete',
          '#title' => $this->options['expose']['label'] ?? $this->t('Node title'),
          '#target_type' => 'node',
          '#default_value' => NULL,
          '#selection_settings' => [
            'match_operator' => 'CONTAINS',
            'target_bundles' => $this->detectBundlesFromView(),
          ],
          '#description' => $this->t('Start typing to search node titles.'),
        ];

        if (!empty($this->value) && is_numeric($this->value)) {
          $node = $this->entityTypeManager->getStorage('node')->load((int) $this->value);
          if ($node) {
            $form[$identifier]['#default_value'] = $node;
          }
        }
        break;

      case 'checkboxes':
        $form[$identifier] = [
          '#type' => 'checkboxes',
          '#title' => $this->options['expose']['label'] ?? $this->t('Node title'),
          '#options' => $options,
          '#default_value' => is_array($this->value) ? $this->value : ($this->value ? [$this->value] : []),
        ];
        break;

      case 'select':
      default:
        $select_options = ['' => $this->t('- Any -')] + $options;
        $form[$identifier] = [
          '#type' => 'select',
          '#title' => $this->options['expose']['label'] ?? $this->t('Node title'),
          '#options' => $select_options,
          '#default_value' => $this->value ?? '',
        ];
        break;
    }

    if (!empty($this->options['expose']['identifier'])) {
      $form[$identifier]['#name'] = $this->options['expose']['identifier'];
    }
  }

  /**
   * @inheritdoc
   */
  public function acceptExposedInput($input) {
    if (empty($this->options['expose'])) {
      return FALSE;
    }
    $identifier = $this->options['expose']['identifier'] ?? ('node_title_select_' . $this->definition['id']);
    if (!isset($input[$identifier]) || $input[$identifier] === '') {
      $this->value = NULL;
      return FALSE;
    }

    $raw = $input[$identifier];
    $widget = $this->options['widget'] ?? 'select';

    if ($widget === 'autocomplete') {
      if (is_string($raw)) {
        if (preg_match('/^\\d+$/', $raw)) {
          $nid = (int) $raw;
        }
        elseif (preg_match('/\\((\\d+)\\)$/', $raw, $m)) {
          $nid = (int) $m[1];
        }
        else {
          $nid = NULL;
        }
        $this->value = $nid;
        return (bool) $nid;
      }
      return FALSE;
    }

    if ($widget === 'checkboxes') {
      $selected = [];
      if (is_array($raw)) {
        foreach ($raw as $k => $v) {
          if ($v !== 0 && $v !== '0' && $v !== '' && $v !== NULL) {
            $selected[] = (int) $k;
          }
        }
      }
      if (empty($selected)) {
        $this->value = NULL;
        return FALSE;
      }
      $this->value = $selected;
      return TRUE;
    }

    if (is_scalar($raw) && $raw !== '') {
      if (is_numeric($raw)) {
        $this->value = (int) $raw;
        return TRUE;
      }
      $this->value = NULL;
      return FALSE;
    }

    $this->value = NULL;
    return FALSE;
  }

  /**
   * @inheritdoc
   */
  public function query() {
    if ($this->value === NULL || $this->value === []) {
      return;
    }
    $table = $this->ensureMyTable();
    $field = 'nid';
    if (is_array($this->value)) {
      $nids = array_map('intval', $this->value);
      if (empty($nids)) {
        return;
      }
      $this->query->addWhere($this->options['group'], "$table.$field", $nids, 'IN');
      return;
    }

    if (is_numeric($this->value)) {
      $this->query->addWhere($this->options['group'], "$table.$field", (int) $this->value, '=');
    }
  }

  /**
   * @inheritdoc
   */
  public function summaryTitle() {
    return $this->t('Node title (select/autocomplete/checkboxes)');
  }

}