(function ($, window, Drupal, once) {
  'use strict';

  Drupal.behaviors.VoiceRecorder = {
    attach: function (context, settings) {
      once('voice_recorder', '.voice-recorder-recording-button-widget', context)
        .forEach(element => {
          const $button = $(element);
          const $fileInput = $button.siblings('input.form-element-voice-recorder-file-upload');

          if ($button.length && $fileInput.length) {
            if ($button.attr('data-upload-displaying') === 'hide_when_js_is_enabled') {
              $fileInput.hide();
            }

            const config = {
              sampleRate: 44100,
              channels: 1,
              bitRate: 128,
              mimeType: 'audio/mpeg',
              maxDuration: parseInt($button.attr('data-max-recording-time')) || 0,
              countdownTime: parseInt($button.attr('data-countdown-time-before-recording')) || 3
            };

            (new VoiceRecorderController($button, $fileInput, config)).setupTriggerEvent();
          }
        });
    }
  };

  class VoiceRecorderController {
    constructor(triggerButton, fileInput, config) {
      this.triggerButton = triggerButton;
      this.fileInput = fileInput;
      this.config = config;
      this.intervals = {};
      this.handleBeforeUnload = this.handleBeforeUnload.bind(this);
    }

    handleBeforeUnload(e) {
      if (this.recorderCore?.recorder) {
        e.preventDefault();
        e.returnValue = 'Recording in progress. Are you sure?';
        return e.returnValue;
      }
    }

    setupTriggerEvent() {
      this.triggerButton.on('click', async (e) => {
        e.preventDefault();
        if (!this.fileInput) {
          console.error("Unable to locate a file input.");
          return;
        }
        if (this.isAnotherRecorderInProgress()) {
          console.info('Only one Voice Recorder instance can run at a time. Please stop the previous before starting a new one.');
          return;
        }

        $('body').addClass('voice-recorder-in-progress');
        window.voiceRecorder = this;

        this.setupUI();
        this.bindEvents();

        const hasAccess = await this.checkMicrophonePermission();
        if (!hasAccess) {
          this.handleError(Drupal.t('Microphone access denied.'));
          return;
        }

        this.setupRecorder();
        await this.handleStart();
      });
    }

    setupUI() {
      const encodedHtml = this.triggerButton.attr('data-widget-html');
      const widgetHtml = $('<div/>').html(encodedHtml).text();
      this.widget = $(widgetHtml).appendTo('body');

      this.elements = {
        status: this.widget.find('.voice-recorder-status-label'),
        duration: this.widget.find('.voice-recorder-duration'),
        closeBtn: this.widget.find('.voice-recorder-close-button'),
        stopBtn: this.widget.find('.voice-recorder-stop-button'),
        pauseBtn: this.widget.find('.voice-recorder-pause-button'),
        resumeBtn: this.widget.find('.voice-recorder-resume-button'),
        restartBtn: this.widget.find('.voice-recorder-restart-button')
      };
      this.handleStateChange('preparing');
    }

    handleStateChange(state, value = '') {
      const stateMap = {
        preparing: {
          text: Drupal.t('Preparing...'),
          visibleButtons: ['closeBtn'],
          enabledButtons: ['closeBtn']
        },
        countdown: {
          text: Drupal.t('Recording in @count...', {'@count': value}),
          visibleButtons: ['closeBtn'],
          enabledButtons: ['closeBtn']
        },
        waitingForRecording: {
          text: Drupal.t('Preparing...'),
          visibleButtons: ['closeBtn'],
          enabledButtons: []
        },
        recording: {
          text: Drupal.t('Recording in progress...'),
          visibleButtons: ['stopBtn', 'pauseBtn', 'restartBtn'],
          enabledButtons: ['stopBtn', 'pauseBtn', 'restartBtn']
        },
        paused: {
          text: Drupal.t('Recording paused'),
          visibleButtons: ['stopBtn', 'resumeBtn', 'restartBtn'],
          enabledButtons: ['stopBtn', 'resumeBtn', 'restartBtn']
        },
        stopping: {
          text: Drupal.t('Stopping recording...'),
          visibleButtons: ['stopBtn'],
          enabledButtons: []
        },
        processing: {
          text: Drupal.t('Processing recording: @progress%', {'@progress': value}),
          visibleButtons: ['stopBtn'],
          enabledButtons: []
        },
        error: {
          text: value,
          visibleButtons: ['closeBtn'],
          enabledButtons: ['closeBtn']
        }
      };

      const stateConfig = stateMap[state];
      if (!stateConfig) {
        return;
      }

      // Update status text.
      this.elements.status.text(stateConfig.text);

      // Make all buttons invisible and disabled by default.
      ['closeBtn', 'stopBtn', 'pauseBtn', 'resumeBtn', 'restartBtn'].forEach(btn => {
        this.elements[btn].hide().prop('disabled', true);
      });

      // Show and enable specific buttons.
      stateConfig.visibleButtons.forEach(btn => {
        this.elements[btn].show();
      });

      stateConfig.enabledButtons.forEach(btn => {
        this.elements[btn].prop('disabled', false);
      });
    }

    handleError(message) {
      this.handleStateChange('error', message);
    }

    bindEvents() {
      this.elements.closeBtn.on('click', () => this.handleClose());
      this.elements.stopBtn.on('click', () => this.handleStop());
      this.elements.pauseBtn.on('click', () => this.handlePause());
      this.elements.resumeBtn.on('click', () => this.handleResume());
      this.elements.restartBtn.on('click', () => this.handleRestart());

      const self = this;
      const elRefFieldLabel = this.widget.find(".voice-recorder-initiator-field-label");
      elRefFieldLabel.click(function(e) {
        e.preventDefault();
        const elementPosition = parseInt(self.fileInput.offset().top);
        let scrollTo = 0;
        if (elementPosition - 100 > 0) {
          scrollTo = elementPosition - 100;
        }

        $('html, body').animate({
          scrollTop: scrollTo,
        }, 1000);
      });

      window.addEventListener('beforeunload', this.handleBeforeUnload);

      navigator.permissions.query({ name: 'microphone' }).then(permissionStatus => {
        permissionStatus.onchange = async () => {
          if (permissionStatus.state === 'denied' && this.recorderCore.durationTracker.getCurrentDuration()) {
            this.handleError(Drupal.t('Microphone access was denied.'));
            await this.handleStop();
          }
        };
      });
    }

    isAnotherRecorderInProgress() {
      return window.voiceRecorder || $(".voice-recorder-floating-widget").length > 0;
    }

    async checkMicrophonePermission() {
      try {
        // The navigator.permissions.query() for microphone doesn't work
        // reliably on mobile browsers, especially Safari and Chrome for iOS.
        // Try to actually get the microphone stream.
        const stream = await navigator.mediaDevices.getUserMedia({
          audio: true,
          video: false
        });

        // If we get here, permission was granted. Clean up the stream.
        stream.getTracks().forEach(track => track.stop());
        return true;
      } catch (err) {
        this.handleError(Drupal.t('Unable to check microphone permissions.'));
        return false;
      }
    }

    handleDurationChange(duration) {
      const totalSeconds = Math.floor(duration / 1000);
      const hours = Math.floor(totalSeconds / 3600);
      const minutes = Math.floor((totalSeconds % 3600) / 60);
      const seconds = totalSeconds % 60;
      const formattedDuration = [hours, minutes, seconds]
        .map(n => n.toString().padStart(2, '0'))
        .join(':');

      this.elements.duration.text(formattedDuration);
    }

    setupRecorder() {
      const durationTracker = new RecordingDurationTracker({
        onDurationChange: this.handleDurationChange.bind(this)
      });

      this.recorderCore = new RecorderCore({
        durationTracker,
        config: this.config,
        callbacks: {
          onStateChange: this.handleStateChange.bind(this),
          onError: this.handleError.bind(this),
          onMaxDurationReached: this.handleMaxDurationReached.bind(this)
        }
      });
    }

    async handleStart() {
      try {
        if (await this.recorderCore.prepareRecorder()) {
          await this.startCountdown();
        }
      } catch (err) {
        console.error('Recording error:', err);
        this.handleError(Drupal.t('Unable to start recording.'));
      }
    }

    async startCountdown() {
      if (!this.config.countdownTime || this.config.countdownTime <= 0) {
        this.handleStateChange('waitingForRecording');
        return await this.recorderCore.startRecording();
      }

      let countdownTime = this.config.countdownTime;
      this.handleStateChange('countdown', countdownTime);

      const doCountdown = () => new Promise((resolve) => {
        clearInterval(this.intervals.countdown);
        this.intervals.countdown = setInterval(() => {
          countdownTime--;

          if (countdownTime > 0) {
            this.handleStateChange('countdown', countdownTime);
          } else {
            this.handleStateChange('waitingForRecording');
            clearInterval(this.intervals.countdown);
            resolve();
          }
        }, 1000);
      });

      return doCountdown().then(async () => {
        return await this.recorderCore.startRecording();
      });
    }

    async handleRestart() {
      clearInterval(this.intervals.countdown);

      if (this.recorderCore) {
        await this.recorderCore.resetRecording();
      }

      await this.handleStart();
    }

    handlePause() {
      this.recorderCore.pauseRecording();
    }

    handleResume() {
      this.recorderCore.resumeRecording();
    }

    async handleStop() {
      clearInterval(this.intervals.countdown);

      if (this.recorderCore) {
        const blob = await this.recorderCore.stopRecording();
        if (blob) {
          this.saveRecording(blob);
        }
      }
      this.cleanup();
    }

    async handleClose() {
      clearInterval(this.intervals.countdown);

      if (this.recorderCore) {
        await this.recorderCore.resetRecording();
      }
      this.cleanup();
    }

    async handleMaxDurationReached() {
      await this.handleStop();
    }

    saveRecording(blob) {
      if (!blob) {
        return;
      }

      const file = new File([blob], this.generateFileName(), {
        type: 'audio/mpeg',
        lastModified: Date.now()
      });

      const dataTransfer = new DataTransfer();
      dataTransfer.items.add(file);
      this.fileInput[0].files = dataTransfer.files;
      this.fileInput.trigger('change');
    }

    generateFileName() {
      const now = new Date();
      const datePart = now.toISOString().slice(0, 10);
      const timePart = now.toTimeString().slice(0, 8).replace(/:/g, "");
      const hash = this.triggerButton.attr('data-upload-random-sequence');
      return `recording_${datePart}_${timePart}_${hash}.mp3`;
    }

    cleanup() {
      window.removeEventListener('beforeunload', this.handleBeforeUnload);
      this.widget.remove();
      $('body').removeClass('voice-recorder-in-progress');
      window.voiceRecorder = null;

      Object.values(this.intervals).forEach(interval => clearInterval(interval));
      this.intervals = {};
    }
  }

  class RecorderCore {
    constructor({ durationTracker, config, callbacks }) {
      this.durationTracker = durationTracker;
      this.config = config;
      this.recorder = null;
      this.mediaStream = null;
      this.audioContext = null;
      this.intervals = {};
      this.state = 'inactive';
      this.callbacks = {
        onStateChange: () => {},
        onError: () => {},
        onMaxDurationReached: () => {},
        ...callbacks
      };
    }

    setState(newState) {
      this.state = newState;
      this.callbacks.onStateChange(newState);
    }

    async prepareRecorder() {
      try {
        const AudioContext = window.AudioContext || window.webkitAudioContext;
        if (!AudioContext || !navigator.mediaDevices) {
          this.callbacks.onError('Browser not supported');
          return false;
        }

        // Some browsers do not support specifying a desired sample rate in
        // AudioContext, so use any sample rate determined by the browser
        // and convert later to desired.
        this.audioContext = new AudioContext();

        this.mediaStream = await navigator.mediaDevices.getUserMedia({
          audio: {
            echoCancellation: true,
            noiseSuppression: true,
            autoGainControl: true,
            channelCount: this.config.channels,
          },
          video: false
        });

        const source = this.audioContext.createMediaStreamSource(this.mediaStream);
        this.recorder = new Recorder(source, {
          numChannels: this.config.channels,
          mimeType: this.config.mimeType
        });
        return true;
      } catch (err) {
        this.mediaStream = null;
        this.audioContext = null;
        this.recorder = null;
        console.error('Recording error:', err);
        this.callbacks.onError('Unable to start recording');
        return false;
      }
    }

    async startRecording() {
      if (!this.recorder) {
        return false;
      }

      this.setState('recording');
      this.recorder.record();
      this.durationTracker.start();
      this.startMaxDurationCheck();
      return true;
    }

    startMaxDurationCheck() {
      if (this.config.maxDuration <= 0) {
        return;
      }

      clearInterval(this.intervals.maxDuration);
      this.intervals.maxDuration = setInterval(() => {
        const currentDuration = this.durationTracker.getCurrentDuration();
        if (currentDuration >= this.config.maxDuration * 1000) {
          clearInterval(this.intervals.maxDuration);
          this.callbacks.onMaxDurationReached();
        }
      }, 100);
    }

    pauseRecording() {
      if (this.recorder?.recording) {
        this.setState('paused');
        clearInterval(this.intervals.maxDuration);
        this.recorder.stop();
        this.durationTracker.pause();
      }
    }

    resumeRecording() {
      if (this.recorder && !this.recorder.recording) {
        this.setState('recording');
        this.startMaxDurationCheck();
        this.recorder.record();
        this.durationTracker.resume();
      }
    }

    async resetRecording() {
      clearInterval(this.intervals.maxDuration);
      this.durationTracker.reset();

      if (this.recorder) {
        this.recorder.stop();
        this.recorder = null;
      }

      if (this.mediaStream) {
        this.mediaStream.getAudioTracks().forEach(track => track.stop());
        this.mediaStream = null;
      }

      if (this.audioContext) {
        try {
          await this.audioContext.suspend();
          await this.audioContext.close();
          this.audioContext = null;
        } catch (err) {
          console.warn('Error closing audio context:', err);
        }
      }
    }

    async stopRecording() {
      clearInterval(this.intervals.maxDuration);

      if (!this.recorder) {
        return null;
      }

      this.setState('stopping');
      this.recorder.stop();
      this.durationTracker.stop();
      this.mediaStream?.getAudioTracks()[0].stop();

      return new Promise((resolve) => {
        this.recorder.getBuffer(async (buffers) => {
          const mp3Data = await this.processMp3(buffers[0]);
          const blob = new Blob(mp3Data, {type: this.config.mimeType});
          resolve(blob);
        });
      });
    }

    async processMp3(buffer) {
      let audioBuffer = buffer;

      // Convert the sample rate if it differs from the requested rate.
      if (this.config.sampleRate !== this.audioContext.sampleRate) {
        audioBuffer = await this.convertSampleRate(buffer, this.audioContext.sampleRate, this.config.sampleRate);
      }

      const mp3encoder = new lamejs.Mp3Encoder(
        this.config.channels,
        this.config.sampleRate,
        this.config.bitRate
      );

      // Convert audio samples to 16-bit PCM format required for MP3 encoding.
      // Audio samples come as float numbers between -1.0 and 1.0.
      // We convert them to 16-bit integers in range from -32768 to 32767.
      const pcmSamples = Int16Array.from(audioBuffer.map(sample => {
        const normalizedSample = Math.max(-1, Math.min(1, sample));
        return normalizedSample < 0 ? normalizedSample * 0x8000 : normalizedSample * 0x7FFF;
      }));

      const mp3Data = [];
      const mp3FrameSize = 1152;
      const continuousEncodeLimit = 30;
      let processedSamplesLength = 0;

      // Encode PCM samples into fixed-size MP3 frames (1152 samples each) for
      // standard compliance. Add silence (zeros) to the last frame if it has
      // fewer samples to ensure compatibility and avoid playback issues.
      while (processedSamplesLength < pcmSamples.length) {
        const samplesToCopy = Math.min(mp3FrameSize, pcmSamples.length - processedSamplesLength);

        // Prepare a silence frame (will be zero-filled).
        const frame = new Int16Array(mp3FrameSize);

        // Copy available samples (rest will be silence).
        frame.set(pcmSamples.subarray(processedSamplesLength, processedSamplesLength + samplesToCopy));

        const encoded = mp3encoder.encodeBuffer(frame);
        if (encoded.length > 0) {
          mp3Data.push(encoded);
        }

        processedSamplesLength += mp3FrameSize;

        // Update progress and allow UI updates periodically.
        if (processedSamplesLength % (mp3FrameSize * continuousEncodeLimit) === 0) {
          const progress = Math.round((processedSamplesLength / pcmSamples.length) * 100);
          this.callbacks.onStateChange('processing', progress);
          await new Promise(resolve => setTimeout(resolve, 0));
        }
      }

      // Add final frame.
      const finalFrame = mp3encoder.flush();
      if (finalFrame.length > 0) {
        mp3Data.push(finalFrame);
      }

      this.callbacks.onStateChange('processing', 100);

      return mp3Data;
    }

    async convertSampleRate(audioData, fromRate, toRate) {
      // Simple linear interpolation for sample rate conversion.
      const ratio = toRate / fromRate;
      const newLength = Math.round(audioData.length * ratio);
      const result = new Float32Array(newLength);

      for (let i = 0; i < newLength; i++) {
        const position = i / ratio;
        const index = Math.floor(position);
        const fraction = position - index;

        if (index + 1 < audioData.length) {
          result[i] = audioData[index] * (1 - fraction) + audioData[index + 1] * fraction;
        } else {
          result[i] = audioData[index];
        }
      }

      return result;
    }
  }

  class RecordingDurationTracker {
    constructor({ onDurationChange }) {
      this.durationTime = 0;
      this.trackingStartTime = 0;
      this.isRunning = false;
      this.intervals = {};
      this.onDurationChange = onDurationChange;
    }

    start() {
      this.durationTime = 0;
      this.isRunning = true;
      this.startTracking();
    }

    pause() {
      if (!this.isRunning) {
        return;
      }
      this.isRunning = false;
      clearInterval(this.intervals.tracking);
      this.updateDuration();
    }

    resume() {
      if (this.isRunning) {
        return;
      }
      this.isRunning = true;
      this.startTracking();
    }

    stop() {
      if (this.isRunning) {
        this.updateDuration();
      }
      clearInterval(this.intervals.tracking);
      this.isRunning = false;
    }

    reset() {
      clearInterval(this.intervals.tracking);
      this.durationTime = 0;
      this.trackingStartTime = 0;
      this.isRunning = false;
      this.onDurationChange(this.durationTime);
    }

    getCurrentDuration() {
      return this.isRunning
        ? this.durationTime + (Date.now() - this.trackingStartTime)
        : this.durationTime;
    }

    updateDuration() {
      this.durationTime += Date.now() - this.trackingStartTime;
    }

    startTracking() {
      this.trackingStartTime = Date.now();
      clearInterval(this.intervals.tracking);
      this.intervals.tracking = setInterval(() => {
        this.onDurationChange(this.getCurrentDuration());
      }, 100);
    }
  }

}(jQuery, window, Drupal, once));
