<?php

namespace Drupal\voice_recorder\Element;

use Drupal\Component\Utility\Crypt;
use Drupal\Component\Utility\Html;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Attribute\FormElement;
use Drupal\file\Element\ManagedFile;

/**
 * Provides an AJAX/progress aware widget for recording and saving a file.
 */
#[FormElement('voice_recorder_file')]
class VoiceFileRecorder extends ManagedFile {

  /**
   * Gets the voice recorder default settings.
   *
   * @return array
   *   The settings.
   */
  public static function getVoiceRecorderDefaultSettings(): array {
    return [
      'max_recording_time' => 0,
      'countdown_time_before_recording' => 0,
      'upload_element_displaying' => 'show',
      'file_information_displaying' => 'filename_link_and_audio',
      'close_recording_label' => 'Close Recording',
      'start_recording_label' => 'Record',
      'stop_recording_label' => 'Stop Recording',
      'pause_recording_label' => 'Pause Recording',
      'resume_recording_label' => 'Resume Recording',
      'restart_recording_label' => 'Restart Recording',
      'recording_for_label' => 'Recording for',
      'initiator_field_label' => '',
    ];
  }

  /**
   * Gets the voice recorder settings for the given element.
   *
   * @return array
   *   The settings.
   */
  public static function getVoiceRecorderSettings($element): array {
    $voice_recorder_settings = $element['#voice_recorder_settings'] ?? [];
    return $voice_recorder_settings + self::getVoiceRecorderDefaultSettings();
  }

  /**
   * Gets the options available for upload element displaying.
   *
   * @return array
   *   An associative array where the keys represent the options and the
   *   values are the labels.
   */
  public static function getUploadElementDisplayingOptions(): array {
    return [
      'show' => t('Show'),
      'hide_when_js_is_enabled' => t('Hide (when JS is enabled)'),
    ];
  }

  /**
   * Gets the options available for displaying file information.
   *
   * @return array
   *   An associative array where the keys represent the options and the
   *   values are the labels.
   */
  public static function getFileInformationDisplayingOptions(): array {
    return [
      'filename_link' => t('Filename link'),
      'audio' => t('Audio'),
      'filename_link_and_audio' => t('Filename link and Audio'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function processManagedFile(&$element, FormStateInterface $form_state, &$complete_form) {
    $element = parent::processManagedFile($element, $form_state, $complete_form);
    $voice_recorder_settings = self::getVoiceRecorderSettings($element);

    if (!empty($element['upload'])) {
      $context = [
        'element' => $element,
        'form_state' => $form_state,
        'complete_form' => $complete_form,
      ];
      \Drupal::moduleHandler()->alter('voice_recorder_file_element_settings', $voice_recorder_settings, $context);

      $recording_floating_widget = [
        '#theme' => 'voice_recorder_floating_widget',
        '#settings' => $voice_recorder_settings,
      ];
      $recording_floating_widget_html = \Drupal::service('renderer')->renderInIsolation($recording_floating_widget);
      $recording_button = [
        '#theme' => 'voice_recorder_recording_button_widget',
        '#settings' => $voice_recorder_settings,
        '#attributes' => [
          'class' => ['voice-recorder-recording-button-widget'],
          'data-max-recording-time' => $voice_recorder_settings['max_recording_time'],
          'data-countdown-time-before-recording' => $voice_recorder_settings['countdown_time_before_recording'],
          'data-upload-displaying' => $voice_recorder_settings['upload_element_displaying'],
          'data-upload-random-sequence' => Crypt::randomBytesBase64(4),
          'data-widget-html' => Html::escape($recording_floating_widget_html),
        ],
        '#weight' => $element['upload']['#weight'],
      ];
      $recording_button_html = \Drupal::service('renderer')->render($recording_button);

      $element['upload']['#suffix'] = $recording_button_html;
      $element['upload']['#attributes']['class'][] = 'form-element-voice-recorder-file-upload';
      $element['upload']['#attributes']['accept'] = 'audio/*';
    }

    $show_file_link = in_array($voice_recorder_settings['file_information_displaying'], ['filename_link', 'filename_link_and_audio']);
    $show_audio = in_array($voice_recorder_settings['file_information_displaying'], ['audio', 'filename_link_and_audio']);
    $fids = $element['#value']['fids'] ?? [];
    if (!empty($fids) && $element['#files']) {
      /** @var \Drupal\file\FileInterface $file */
      foreach ($element['#files'] as $delta => $file) {
        $file_item_renderable = [
          '#type' => 'container',
          '#file' => $file,
        ];
        if ($show_file_link) {
          $file_item_renderable['file_link'] = [
            '#theme' => 'file_link',
            '#file' => $file,
          ];
        }
        if ($show_audio) {
          $file_item_renderable['audio'] = [
            '#type' => 'html_tag',
            '#tag' => 'audio',
            '#attributes' => [
              'controls' => 'controls',
            ],
            '#prefix' => '<div>',
            '#suffix' => '</div>',
            'source' => [
              '#type' => 'html_tag',
              '#tag' => 'source',
              '#attributes' => [
                'src' => \Drupal::service('file_url_generator')->generate($file->getFileUri())->toString(),
                'type' => $file->getMimeType(),
              ],
            ],
          ];
        }
        if ($element['#multiple']) {
          $element['file_' . $delta]['selected'] = [
            '#type' => 'checkbox',
            '#title' => \Drupal::service('renderer')->renderInIsolation($file_item_renderable),
          ];
        }
        else {
          $element['file_' . $delta]['filename'] = $file_item_renderable + ['#weight' => -10];
        }
      }
    }

    return $element;
  }

}
