<?php

namespace Drupal\voice_recorder\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\file\Plugin\Field\FieldWidget\FileWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\voice_recorder\Element\VoiceFileRecorder;

/**
 * A recorder widget for voice file fields.
 */
#[FieldWidget(
  id: 'file_voice_recorder',
  label: new TranslatableMarkup('Voice Recorder File'),
  field_types: ['file'],
)]
class VoiceRecorderWidget extends FileWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return VoiceFileRecorder::getVoiceRecorderDefaultSettings() + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $default_settings = self::defaultSettings();
    $element = parent::settingsForm($form, $form_state);
    $element['max_recording_time'] = [
      '#type' => 'number',
      '#size' => 10,
      '#title' => $this->t('Maximum Recording Time (seconds)'),
      '#default_value' => $this->getSetting('max_recording_time') ?? $default_settings['max_recording_time'],
      '#min' => 0,
      '#description' => $this->t('The maximum recording time allowed (in seconds). Use 0 for unlimited time.'),
      '#weight' => 10,
      '#required' => TRUE,
    ];
    $element['countdown_time_before_recording'] = [
      '#type' => 'number',
      '#size' => 10,
      '#title' => $this->t('Countdown time before recording (seconds)'),
      '#default_value' => $this->getSetting('countdown_time_before_recording') ?? $default_settings['countdown_time_before_recording'],
      '#min' => 0,
      '#description' => $this->t('The number of seconds to display a countdown before recording starts.'),
      '#weight' => 10,
      '#required' => TRUE,
    ];
    $element['upload_element_displaying'] = [
      '#type' => 'select',
      '#title' => $this->t('Upload Element Visibility'),
      '#default_value' => $this->getSetting('upload_element_displaying') ?? $default_settings['upload_element_displaying'],
      '#description' => $this->t('After recording, the audio file will be attached to the upload file element. The "hide" option will hide it at the JavaScript level. When JavaScript is disabled on the site, the upload file element will remain visible.'),
      '#options' => VoiceFileRecorder::getUploadElementDisplayingOptions(),
      '#weight' => 10,
      '#required' => TRUE,
    ];
    $element['file_information_displaying'] = [
      '#type' => 'select',
      '#title' => $this->t('File Information Displaying'),
      '#default_value' => $this->getSetting('file_information_displaying') ?? $default_settings['file_information_displaying'],
      '#description' => $this->t('Choose how the file information should be rendered for the uploaded file after AJAX refresh.'),
      '#options' => VoiceFileRecorder::getFileInformationDisplayingOptions(),
      '#weight' => 10,
      '#required' => TRUE,
    ];
    $element['start_recording_label'] = [
      '#type' => 'textfield',
      '#size' => 10,
      '#title' => $this->t('Start recording button label'),
      '#default_value' => $this->getSetting('start_recording_label') ?? $default_settings['start_recording_label'],
      '#description' => $this->t('The label for the button that starts recording and displays the recording floating widget.'),
      '#weight' => 10,
      '#required' => TRUE,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    if (empty($this->getSetting('max_recording_time'))) {
      $summary[] = $this->t('Max recording time: unlimited');
    }
    else {
      $summary[] = $this->t('Max recording time: @value seconds', ['@value' => $this->getSetting('max_recording_time')]);
    }
    if (empty($this->getSetting('countdown_time_before_recording'))) {
      $summary[] = $this->t('Countdown before recording: no');
    }
    else {
      $summary[] = $this->t('Countdown before recording: @value seconds', ['@value' => $this->getSetting('countdown_time_before_recording')]);
    }
    $summary[] = $this->t('Upload element displaying: @value', ['@value' => VoiceFileRecorder::getUploadElementDisplayingOptions()[$this->getSetting('upload_element_displaying')]]);
    $summary[] = $this->t('File information displaying: @value', ['@value' => VoiceFileRecorder::getFileInformationDisplayingOptions()[$this->getSetting('file_information_displaying')]]);
    $summary[] = $this->t('Start recording label: @value', ['@value' => $this->getSetting('start_recording_label')]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   *
   * This method was copied from the parent class to add a new upload validator
   * and pass it to the description help renderer. A simple way to continue
   * using the default file field.
   *
   * Changes from base:
   * Element info and type are set to the custom element 'voice_recorder_file'.
   * Settings from the field widget are passed to '#voice_recorder_settings'.
   * A new upload validator is added.
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $field_settings = $this->getFieldSettings();

    // The field settings include defaults for the field type. However, this
    // widget is a base class for other widgets (e.g., ImageWidget) that may act
    // on field types without these expected settings.
    $field_settings += [
      'display_default' => NULL,
      'display_field' => NULL,
      'description_field' => NULL,
    ];

    $cardinality = $this->fieldDefinition->getFieldStorageDefinition()->getCardinality();
    $defaults = [
      'fids' => [],
      'display' => (bool) $field_settings['display_default'],
      'description' => '',
    ];

    // Essentially we use the voice_recorder_file type, extended with some
    // enhancements.
    $element_info = $this->elementInfo->getInfo('voice_recorder_file');
    $element += [
      '#type' => 'voice_recorder_file',
      '#upload_location' => $items[$delta]->getUploadLocation(),
      '#upload_validators' => $items[$delta]->getUploadValidators(),
      '#value_callback' => [static::class, 'value'],
      '#process' => array_merge($element_info['#process'], [[static::class, 'process']]),
      '#progress_indicator' => $this->getSetting('progress_indicator'),
      // Allows this field to return an array instead of a single value.
      '#extended' => TRUE,
      // Add properties needed by value() and process() methods.
      '#field_name' => $this->fieldDefinition->getName(),
      '#entity_type' => $items->getEntity()->getEntityTypeId(),
      '#display_field' => (bool) $field_settings['display_field'],
      '#display_default' => $field_settings['display_default'],
      '#description_field' => $field_settings['description_field'],
      '#cardinality' => $cardinality,
    ];

    // Add own additional validators.
    $max_recording_time = $this->getSetting('max_recording_time');
    if ($max_recording_time && $max_recording_time > 0) {
      $element['#upload_validators']['VoiceRecorderFileAudioDurationLimit'] = [
        'durationLimit' => $max_recording_time,
      ];
    }

    // Add own settings related to voice_recorder_file type.
    $element['#voice_recorder_settings'] = [
      'initiator_field_label' => $element['#title'] ?? '',
      'max_recording_time' => $this->getSetting('max_recording_time'),
      'countdown_time_before_recording' => $this->getSetting('countdown_time_before_recording'),
      'upload_element_displaying' => $this->getSetting('upload_element_displaying'),
      'file_information_displaying' => $this->getSetting('file_information_displaying'),
      'close_recording_label' => $this->getSetting('close_recording_label'),
      'start_recording_label' => $this->getSetting('start_recording_label'),
      'stop_recording_label' => $this->getSetting('stop_recording_label'),
      'pause_recording_label' => $this->getSetting('pause_recording_label'),
      'resume_recording_label' => $this->getSetting('resume_recording_label'),
      'restart_recording_label' => $this->getSetting('restart_recording_label'),
      'recording_for_label' => $this->getSetting('recording_for_label'),
    ];

    $element['#weight'] = $delta;

    // Field stores FID value in a single mode, so we need to transform it for
    // form element to recognize it correctly.
    if (!isset($items[$delta]->fids) && isset($items[$delta]->target_id)) {
      $items[$delta]->fids = [$items[$delta]->target_id];
    }
    $element['#default_value'] = $items[$delta]->getValue() + $defaults;

    $default_fids = $element['#extended'] ? $element['#default_value']['fids'] : $element['#default_value'];
    if (empty($default_fids)) {
      $file_upload_help = [
        '#theme' => 'file_upload_help',
        '#description' => $element['#description'],
        '#upload_validators' => $element['#upload_validators'],
        '#cardinality' => $cardinality,
      ];
      $element['#description'] = \Drupal::service('renderer')->renderInIsolation($file_upload_help);
      $element['#multiple'] = $cardinality != 1 ? TRUE : FALSE;
      if ($cardinality != 1 && $cardinality != -1) {
        $element['#element_validate'] = [[static::class, 'validateMultipleCount']];
      }
    }

    return $element;
  }

}
