<?php

namespace Drupal\vote_anon\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Database\Connection;
use Drupal\user\UserStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Handles AJAX voting for single node voting.
 */
class SingleNodeVoteAjaxController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The user storage.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  protected UserStorageInterface $userStorage;

  /**
   * Constructs the controller.
   */
  public function __construct(
    Connection $database,
    UserStorageInterface $user_storage
  ) {
    $this->database = $database;
    $this->userStorage = $user_storage;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database'),
      $container->get('entity_type.manager')->getStorage('user')
    );
  }

  /**
   * AJAX callback for single-node voting.
   */
  public function renderSingleNodeVoteLinkRenderable(
    int $node,
    string $nojs,
    Request $request
  ): AjaxResponse {

    $config = $this->config('vote_anon.voteconfiguration');
    $already_vote = $config->get('warning_for_duplicate_voting');
    $new_vote = $config->get('message_after_voting');
    $disable_vote_link = (bool) $config->get('disable_vote_link');

    $session_id = $request->getSession()->getId();

    // Get current user ID using global currentUser service.
    $uid = \Drupal::currentUser()->id();
    $uuid = '';

    if ($uid) {
      $user = $this->userStorage->load($uid);
      $uuid = $user ? $user->uuid() : '';
    }

    $output = '';

    if ($nojs === 'ajax') {

      // Check if the user has already voted for this node.
      $id = $this->database->select('vote_anon', 'va')
        ->fields('va', ['id'])
        ->condition('entity_id', $node)
        ->condition('session_id', $session_id)
        ->execute()
        ->fetchField();

      if (!$id) {

        // Insert vote record.
        $this->database->insert('vote_anon')->fields([
          'entity_type' => 'node',
          'uid' => $uid,
          'uuid' => $uuid,
          'entity_id' => $node,
          'session_id' => $session_id,
          'created' => \Drupal::time()->getRequestTime(),
        ])->execute();

        // Update or insert vote count.
        $vote_id = $this->database->select('vote_anon_counts', 'vac')
          ->fields('vac', ['vote_id'])
          ->condition('entity_id', $node)
          ->execute()
          ->fetchField();

        if ($vote_id) {
          $this->database->update('vote_anon_counts')
            ->expression('count', 'count + 1')
            ->condition('vote_id', $vote_id)
            ->condition('entity_id', $node)
            ->execute();
        }
        else {
          $this->database->insert('vote_anon_counts')->fields([
            'entity_type' => 'node',
            'count' => 1,
            'entity_id' => $node,
            'last_updated' => \Drupal::time()->getRequestTime(),
          ])->execute();
        }

        $output = '<div id="votedestinationdiv' . $node . '" class="ajax-message">'
          . $new_vote . '</div>';
      }
      else {
        $output = '<div id="votedestinationdiv' . $node . '" class="ajax-message">'
          . $already_vote . '</div>';
      }
    }

    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand("#votedestinationdiv{$node}", $output));

    if ($disable_vote_link) {
      $response->addCommand(
        new InvokeCommand(NULL, 'disableVoteLinks', [(string) $node])
      );
    }

    return $response;
  }

}
