<?php

namespace Drupal\vote_anon\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Database\Connection;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\Request;

/**
 * Handles AJAX voting (global voting).
 */
class VoteAjaxController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * Constructs VoteAjaxController.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('database')
    );
  }

  /**
   * AJAX callback for voting.
   */
  public function renderVoteLinkRenderable(
    int $node,
    string $nojs = 'ajax',
    Request $request
  ): AjaxResponse {

    $config = $this->config('vote_anon.voteconfiguration');
    $already_vote = $config->get('warning_for_duplicate_voting');
    $new_vote = $config->get('message_after_voting');
    $cookie_name = $config->get('voting_cookie');
    $disable_vote_link = (bool) $config->get('disable_vote_link');

    $output = '';

    if ($nojs === 'ajax') {

      // Check cookie using Symfony request.
      if (!$request->cookies->has($cookie_name)) {

        $vote_id = $this->database->select('vote_anon_counts', 'vac')
          ->fields('vac', ['vote_id'])
          ->condition('entity_id', $node)
          ->execute()
          ->fetchField();

        if ($vote_id) {
          $this->database->update('vote_anon_counts')
            ->expression('count', 'count + 1')
            ->condition('vote_id', $vote_id)
            ->condition('entity_id', $node)
            ->execute();
        }
        else {
          $this->database->insert('vote_anon_counts')->fields([
            'entity_type' => 'node',
            'count' => 1,
            'entity_id' => $node,
            'last_updated' => \Drupal::time()->getRequestTime(),
          ])->execute();
        }

        $output = '<div id="votedestinationdiv' . $node . '" class="ajax-message">'
          . $new_vote . '</div>';

        $response = new AjaxResponse();
        $response->headers->setCookie(
          new Cookie(
            $cookie_name,
            '1',
            strtotime('+30 days'),
            '/'
          )
        );
      }
      else {
        $output = '<div id="votedestinationdiv' . $node . '" class="ajax-message">'
          . $already_vote . '</div>';

        $response = new AjaxResponse();
      }

      $response->addCommand(
        new ReplaceCommand("#votedestinationdiv{$node}", $output)
      );

      // Disable vote link if configured.
      if ($disable_vote_link) {
        $response->addCommand(
          new InvokeCommand(NULL, 'disableVoteLinks', ['/'])
        );
      }

      return $response;
    }

    // Fallback (should never hit).
    return new AjaxResponse();
  }

  /**
   * Displays vote count listing (non-Views fallback).
   */
  public function voteCount(): array {
    $items = [];

    $query = $this->database->select('vote_anon_counts', 'vac');
    $query->fields('vac', ['entity_id', 'count']);
    $query->join('node_field_data', 'n', 'vac.entity_id = n.nid');
    $query->fields('n', ['title']);

    $results = $query->execute()->fetchAll();

    foreach ($results as $result) {
      $items[] = $this->t('@title – Total Votes: @count', [
        '@title' => $result->title,
        '@count' => $result->count,
      ]);
    }

    return [
      '#title' => $this->t('Voting Details'),
      '#theme' => 'item_list',
      '#items' => $items,
      '#cache' => ['max-age' => 0],
    ];
  }

}
