<?php

namespace Drupal\vote_anon\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Vote Anonymous configuration form.
 */
class VoteConfigurationForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs the form.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['vote_anon.voteconfiguration'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'vote_configuration_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('vote_anon.voteconfiguration');

    // Load node types using Entity API (D10/D11 safe).
    $node_types = $this->entityTypeManager
      ->getStorage('node_type')
      ->loadMultiple();

    $options = ['' => $this->t('- None -')];
    foreach ($node_types as $node_type) {
      $options[$node_type->id()] = $node_type->label();
    }

    $form['message_after_voting'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Message After Voting'),
      '#description' => $this->t('Show a confirmation message after successful voting.'),
      '#maxlength' => 64,
      '#default_value' => $config->get('message_after_voting') ?? $this->t('Thank you for your vote.'),
    ];

    $form['warning_for_duplicate_voting'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Warning for Duplicate Votes'),
      '#description' => $this->t('Show a warning message that you have already voted.'),
      '#maxlength' => 64,
      '#default_value' => $config->get('warning_for_duplicate_voting') ?? $this->t('You have already submitted a vote.'),
    ];

    $form['voting_cookie'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Voting Cookie Name'),
      '#description' => $this->t('Cookie used to track anonymous voting.'),
      '#maxlength' => 64,
      '#default_value' => $config->get('voting_cookie') ?? 'VOTEANON',
      '#required' => TRUE,
    ];

    $form['single_node_voting'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Single Node Voting'),
      '#default_value' => (bool) $config->get('single_node_voting'),
    ];

    $form['disable_vote_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable Link After Vote'),
      '#default_value' => (bool) $config->get('disable_vote_link'),
    ];

    $form['vote_content_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Content Type'),
      '#options' => $options,
      '#default_value' => $config->get('vote_content_type') ?? '',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    $this->config('vote_anon.voteconfiguration')
      ->set('message_after_voting', $form_state->getValue('message_after_voting'))
      ->set('warning_for_duplicate_voting', $form_state->getValue('warning_for_duplicate_voting'))
      ->set('voting_cookie', $form_state->getValue('voting_cookie'))
      ->set('single_node_voting', (bool) $form_state->getValue('single_node_voting'))
      ->set('disable_vote_link', (bool) $form_state->getValue('disable_vote_link'))
      ->set('vote_content_type', $form_state->getValue('vote_content_type'))
      ->save();
  }

}
