<?php

namespace Drupal\ip_login_block\EventSubscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\IpUtils;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Drupal\Core\Config\ConfigFactoryInterface;

class IpLoginBlockSubscriber implements EventSubscriberInterface {

  protected $configFactory;

  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  public static function getSubscribedEvents() {
    $events[KernelEvents::REQUEST][] = ['checkAccess', 20];
    return $events;
  }

  public function checkAccess(RequestEvent $event) {
    $route_match = \Drupal::routeMatch();
    $route_name = $route_match->getRouteName();

    // Check if the current route is the user login page.
    if ($route_name === 'user.login') {
      $config = $this->configFactory->get('ip_login_block.settings');
      if ($config->get('enable_vpn_login')) {
        $request = $event->getRequest();
        $ip_reference_method = $config->get('ip_reference_method');
        $ips = $this->getIpFromRequest($request, $ip_reference_method);
        $allow_ips = $config->get('allow_ips');
        $response_type = $config->get('response_type');

        $block = TRUE;
        foreach ($ips as $ip) {
          if (IpUtils::checkIp($ip, $allow_ips)) {
            $block = FALSE;
            break;
          }
        }

        if ($block) {
          if ($response_type === '404') {
            throw new NotFoundHttpException();
          } else {
            throw new AccessDeniedHttpException();
          }
        }
      }
    }
  }

  /**
   * Gets the IP address from the request based on the selected method.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request.
   * @param string $method
   *   The method to use for getting the IP ('direct_ip' or 'x_forwarded_for').
   * @return string
   *   The IP address.
   */
  protected function getIpFromRequest($request, $method) {
    if ($method === 'x_forwarded_for' && $request->headers->has('X-Forwarded-For')) {
      $ips = explode(',', $request->headers->get('X-Forwarded-For'));

      foreach ($ips as $key => $ip) {
        $parts = explode(":", $ip);
        $ips[$key] = $parts[0];
      }

      return $ips;
    } else {
      return [$request->getClientIp()];
    }
  }
}
