<?php

namespace Drupal\vrssfa\Routing;

use Drupal\Core\Routing\RouteSubscriberBase;
use Symfony\Component\Routing\RouteCollection;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Routing\RouteProviderInterface;
use Psr\Log\LoggerInterface;

/**
 * Listens to the dynamic route events.
 */
class RouteSubscriberViewsrssFeed extends RouteSubscriberBase {

  /**
   * Configuration Factory.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected $configFactory;
  /**
   * The route provider.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected $routeProvider;

  /**
   * Logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructor.
   */
  public function __construct(ConfigFactory $configFactory, RouteProviderInterface $route_provider, LoggerInterface $logger) {
    $this->configFactory = $configFactory;
    $this->routeProvider = $route_provider;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  protected function alterRoutes(RouteCollection $collection) {
    // Get configuration safely.
    $config = $this->configFactory->get('views_rss_authentication.settings');
    if (!$config) {
      $this->logger->warning('Missing configuration: views_rss_authentication.settings');
      return;
    }

    $config_data = $config->get('view_rss_feed_url');
    $authentication_type = $config->get('authentication_type');

    if ($config_data === null || $config_data === '' || $authentication_type === null || $authentication_type === '') {
      $this->logger->warning('Missing RSS feed URLs or authentication type in configuration.');
      return;
    }

    $urls = array_filter(array_map('trim', explode(PHP_EOL, $config_data)));
    foreach ($urls as $path) {
      if ($path === null || $path === '') {
        continue;
      }
      try {
        $found_routes = $this->routeProvider->getRoutesByPattern($path);
        if ($found_routes === null || (is_countable($found_routes) && count($found_routes) === 0)) {
          $this->logger->notice('No routes found for RSS feed path: @path', ['@path' => $path]);
          continue;
        }
        foreach ($found_routes as $route_name => $route_object) {
          $route = $collection->get($route_name);
          if ($route) {
            $route->setOption('_auth', [$authentication_type]);
            $route->setRequirement('_user_is_logged_in', 'TRUE');
            $collection->add($route_name, $route);
          } else {
            $this->logger->warning('Route not found in collection: @route', ['@route' => $route_name]);
          }
        }
      } catch (\Exception $e) {
        $this->logger->error('Error altering route for path @path: @message', [
          '@path' => $path,
          '@message' => $e->getMessage(),
        ]);
      }
    }
  }

}
