<?php

namespace Drupal\vvjs\Twig;

use Twig\Extension\AbstractExtension;
use Twig\Markup;
use Twig\TwigFilter;

/**
 * Class VVJSTwigExtension.
 *
 * Provides a custom Twig extension for marking HTML content as safe.
 *
 * This filter should only be used on HTML that is known to be safe,
 * such as content generated by trusted sources.
 * Avoid using this filter on user-generated content without proper sanitization
 *
 * @package Drupal\vvjs\Twig
 */
class VVJSTwigExtension extends AbstractExtension {

  /**
   * {@inheritdoc}
   */
  public function getFilters(): array {
    return [
      new TwigFilter('safe_html', $this->safeHtml(...), ['is_safe' => ['html']]),
    ];
  }

  /**
   * Marks the provided HTML string as safe.
   *
   * This method should only be used for HTML content that is
   * guaranteed to be safe and not influenced by user input.
   *
   * @param string $string
   *   The string to mark as safe.
   *
   * @return \Twig\Markup
   *   The safe HTML string.
   */
  public function safeHtml(string $string): Markup {
    $decoded_string = html_entity_decode($string, ENT_QUOTES, 'UTF-8');
    return new Markup($decoded_string, 'UTF-8');
  }

}
