<?php

namespace Drupal\vwo\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\vwo\VwoHelp;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * VWO Settings form.
 */
class Settings extends ConfigFormBase {

  /**
   * The VWO help service.
   *
   * @var \Drupal\vwo\VwoHelp
   */
  protected $vwoHelp;

  /**
   * Constructs a Settings object.
   *
   * @param \Drupal\vwo\VwoHelp $vwo_help
   *   The VWO help service.
   */
  public function __construct(VwoHelp $vwo_help) {
    $this->vwoHelp = $vwo_help;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('vwo.help')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'vwo_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'vwo.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#attached']['library'][] = 'vwo/admin';

    $config = $this->config('vwo.settings');

    // Add the VWO help settings block at the top.
    $form['vwo_help_settings'] = [
      '#type' => 'markup',
      '#markup' => $this->vwoHelp->getSettingsUpdatedHelp(),
      '#weight' => -100,
    ];

    $id = $config->get('id');
    $form['id_fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Account'),
    ];
    $form['id_fieldset']['id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('VWO Account ID'),
      '#description' => $this->t('You can find this in your VWO dashboard under Account Settings.'),
      '#size' => 15,
      '#maxlength' => 20,
      '#required' => TRUE,
      '#default_value' => ($id == NULL) ? 'NONE' : $id,
    ];

    $form['synchtype_fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Asynchronous/ Synchronous loading'),
    ];
    $form['synchtype_fieldset']['synchtype'] = [
      '#type' => 'radios',
      '#title' => $this->t('SmartCode Loading Type'),
      '#description' => implode('', [
        '<p>',
        $this->t('Asynchronous loading is now the default. Please see <a target="_blank" href="https://vwo.com/blog/asynchronous-code/">https://vwo.com/blog/asynchronous-code/</a> for more details.'),
        '</p><p>',
        $this->t('The Asynchronous version of VWO code reduces page load time as the VWO code is downloaded in parallel to site code. It also ensures that your site is never slowed down even if the VWO servers are inaccessible.'),
        '</p><p>',
        $this->t('VWO have extensively tested the asynchronous code across variety of browsers (including IE7) and it works perfectly.'),
        '</p><p><strong>',
        $this->t('NB: Due to change in D8 inline javascript, synchronous code is not implemented.'),
        '</strong> ',
        $this->t('See changelog for more details.'),
        '</p>',
      ]),
      '#options' => [
        'async' => $this->t('Asynchronous (default)'),
        'sync' => $this->t('Synchronous'),
      ],
      '#required' => TRUE,
      '#default_value' => $config->get('loading.type'),
    ];

    $form['advanced'] = [
      '#type' => 'details',
      '#open' => FALSE,
      '#title' => $this->t('Advanced Asynchronous Settings'),
      '#description' => implode('', [
        '<p>',
        $this->t('These settings are only used when Asynchronous loading mode is selected above.'),
        '</p><p>',
        '</p>',
      ]),
      '#states' => [
        'visible' => [
          ':input[name=synchtype]' => ['value' => 'async'],
        ],
      ],
    ];

    $form['advanced']['asynctolsettings'] = [
      '#type' => 'number',
      '#title' => $this->t('Settings Tolerance (ms)'),
      '#description' => $this->t('Maximum wait time for VWO settings to be applied. Default: 2000 ms.'),
      '#size' => 10,
      '#min' => 2000,
      '#max' => 9999,
      '#required' => TRUE,
      '#default_value' => $config->get('loading.timeout.settings'),
    ];
    // Add the VWO help report block at the bottom.
    $form['vwo_help_report'] = [
      '#type' => 'markup',
      '#markup' => $this->vwoHelp->getReportHelp(),
      '#weight' => 100,
    ];

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Save configuration'),
        '#button_type' => 'primary',
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {

    // VWO ID Was not set to "number" field to allow for setting it to "NONE",
    // and so much be manually validated.
    $vwoid = $form_state->getValue('id');
    if (!preg_match('/^\d+$/', $vwoid) && $vwoid != 'NONE') {
      $form_state->setErrorByName('id',
        $this->t('Your VWO ID must be numeric (or set to "NONE" to disable). If you have having issues locating it, please use the <a href=":parse_url">Extract Account Id tool</a>.', [
          ':parse_url' => Url::fromRoute('vwo.settings.vwoid')->toString(),
        ])
      );
    }

    // Translate that NONE into NULL for Config API.
    if ($vwoid == 'NONE') {
      $form_state->setValue('id', NULL);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    // Grab the editable configuration.
    $config = $this->config('vwo.settings');

    // Set each of the configuration values.
    $field_key_config_map = [
      'id' => 'id',
      'synchtype' => 'loading.type',
      'asynctolsettings' => 'loading.timeout.settings',
    ];
    foreach ($field_key_config_map as $field_key => $config_key) {
      $config->set($config_key, $form_state->getValue($field_key));
    }

    // Commit saved configuration.
    $config->save();

    $this->messenger()->addMessage($this->t('VWO settings have been saved.'));
  }

}
