<?php

namespace Drupal\wa\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Flood\FloodInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\wa\Service\PasskeyLabeler;
use Drupal\wa\Service\WebAuthnService;
use Webauthn\PublicKeyCredentialCreationOptions;
use Webauthn\PublicKeyCredentialParameters;
use Webauthn\PublicKeyCredentialRpEntity;
use Webauthn\PublicKeyCredentialUserEntity;
use Webauthn\PublicKeyCredentialDescriptor;
use Webauthn\PublicKeyCredentialRequestOptions;
use Webauthn\PublicKeyCredentialSource;
use Webauthn\TrustPath\EmptyTrustPath;
use Symfony\Component\Uid\Uuid;
use ParagonIE\ConstantTime\Base64UrlSafe;
use Drupal\user\UserInterface;

/**
 * Controller for Passkey registration, authentication, and management.
 *
 * This controller handles the routes for:
 * - Listing and managing user passkeys.
 * - Generating options for WebAuthn registration and authentication.
 * - Verifying WebAuthn responses for registration and login.
 */
class PasskeyController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The passkey labeler service.
   *
   * @var \Drupal\wa\Service\PasskeyLabeler
   */
  protected $passkeyLabeler;

  /**
   * The WebAuthn service.
   *
   * @var \Drupal\wa\Service\WebAuthnService
   */
  protected $webAuthnService;

  /**
   * The CSRF token generator.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected $csrfToken;

  /**
   * The flood service.
   *
   * @var \Drupal\Core\Flood\FloodInterface
   */
  protected $flood;

  /**
   * Constructs a PasskeyController object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\wa\Service\PasskeyLabeler $passkey_labeler
   *   The passkey labeler service.
   * @param \Drupal\wa\Service\WebAuthnService $web_authn_service
   *   The WebAuthn service.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
   *   The CSRF token generator.
   * @param \Drupal\Core\Flood\FloodInterface $flood
   *   The flood service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, Connection $database, LoggerChannelFactoryInterface $logger_factory, TimeInterface $time, RequestStack $request_stack, PasskeyLabeler $passkey_labeler, WebAuthnService $web_authn_service, CsrfTokenGenerator $csrf_token, FloodInterface $flood) {
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
    $this->loggerFactory = $logger_factory;
    $this->time = $time;
    $this->requestStack = $request_stack;
    $this->passkeyLabeler = $passkey_labeler;
    $this->webAuthnService = $web_authn_service;
    $this->csrfToken = $csrf_token;
    $this->flood = $flood;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('database'),
      $container->get('logger.factory'),
      $container->get('datetime.time'),
      $container->get('request_stack'),
      $container->get('wa.passkey_labeler'),
      $container->get('wa.webauthn'),
      $container->get('csrf_token'),
      $container->get('flood')
    );
  }

  /**
   * Checks if the current user has an allowed role for passkey operations.
   *
   * @return bool
   *   TRUE if the user has an allowed role or if no role restrictions exist,
   *   FALSE otherwise.
   */
  protected function currentUserHasAllowedRole(): bool {
    $config = \Drupal::config('wa.settings');
    $allowed_roles = $config->get('allowed_roles') ?? [];
    $allowed_roles = array_filter($allowed_roles);

    // If no role restrictions, allow all authenticated users.
    if (empty($allowed_roles)) {
      return TRUE;
    }

    $user_roles = $this->currentUser()->getRoles();
    return !empty(array_intersect($user_roles, $allowed_roles));
  }

  /**
   * Renders the Passkey management page.
   */
  public function managePasskeys(UserInterface $user) {
    $delete_path = ltrim(Url::fromRoute('wa.passkey_delete')->getInternalPath(), '/');
    $delete_token = $this->csrfToken->get($delete_path);
    $register_path = ltrim(Url::fromRoute('wa.register_options')->getInternalPath(), '/');
    $register_token = $this->csrfToken->get($register_path);
    $login_path = ltrim(Url::fromRoute('wa.login_options')->getInternalPath(), '/');
    $login_token = $this->csrfToken->get($login_path);

    $build = [
      '#type' => 'container',
      '#attributes' => ['class' => ['passkey-management']],
      '#attached' => [
        'library' => ['wa/passkey'],
        'drupalSettings' => [
          'wa' => [
            'passkeyDeleteToken' => $delete_token,
            'registerOptionsToken' => $register_token,
            'loginOptionsToken' => $login_token,
            'routes' => [
              'passkeyDelete' => Url::fromRoute('wa.passkey_delete')->toString(),
              'registerOptions' => Url::fromRoute('wa.register_options')->toString(),
              'registerVerify' => Url::fromRoute('wa.register_verify')->toString(),
              'loginOptions' => Url::fromRoute('wa.login_options')->toString(),
              'loginVerify' => Url::fromRoute('wa.login_verify')->toString(),
            ],
          ],
        ],
      ],
    ];

    $build['title'] = [
      '#markup' => '<h2>' . $this->t('Manage Passkeys') . '</h2>',
    ];

    $build['description'] = [
      '#markup' => '<p>' . $this->t('Passkeys allow you to log in securely without a password.') . '</p>',
    ];

    // List existing passkeys.
    $header = [
      'authenticator' => $this->t('Authenticator'),
      'created' => $this->t('Created'),
      'last_used' => $this->t('Last Used'),
      'actions' => $this->t('Actions'),
    ];

    $rows = [];
    $query = $this->database->select('wa', 'c')
      ->fields('c', ['credential_id', 'created', 'last_used', 'aaguid'])
      ->condition('uid', $user->id())
      ->execute();

    foreach ($query as $record) {
      $credentialIdEncoded = base64_encode($record->credential_id);
      $rows[] = [
        $this->passkeyLabeler->getKeyTypeFromRecord($record),
        date('Y-m-d H:i', $record->created),
        date('Y-m-d H:i', $record->last_used),
        [
          'data' => [
            '#type' => 'button',
            '#value' => $this->t('Delete'),
            '#attributes' => [
              'class' => ['button', 'button--danger', 'passkey-delete'],
              'data-credential-id' => $credentialIdEncoded,
            ],
          ],
        ],
      ];
    }

    $build['passkey_management'] = [
      '#type' => 'details',
      '#title' => $this->t('Your Passkeys'),
      '#open' => TRUE,
    ];

    $canRegister = (int) $this->currentUser()->id() === (int) $user->id();

    if (empty($rows)) {
      if ($canRegister) {
        $build['passkey_management']['add_button'] = [
          '#type' => 'button',
          '#value' => $this->t('Add New Passkey'),
          '#attributes' => [
            'id' => 'register-passkey',
            'class' => ['button', 'button--primary'],
          ],
        ];
      }
      $build['passkey_management']['table'] = [
        '#markup' => '<p>' . $this->t('No passkeys registered yet.') . '</p>',
      ];
    }
    else {
      $build['passkey_management']['table'] = [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No passkeys registered yet.'),
      ];
      if ($canRegister) {
        $build['passkey_management']['add_button'] = [
          '#type' => 'button',
          '#value' => $this->t('Add New Passkey'),
          '#attributes' => [
            'id' => 'register-passkey',
            'class' => ['button', 'button--primary'],
          ],
        ];
      }
    }

    $build['passkey_management']['status'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'passkey-status'],
    ];

    return $build;
  }

  /**
   * Provides registration options for passkey creation.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The registration options response.
   */
  public function registerOptions(Request $request) {
    $token = $request->headers->get('X-CSRF-Token');
    $path = ltrim($request->getPathInfo(), '/');

    $config = \Drupal::config('wa.settings');
    if ($config->get('enable_passkey_login') === FALSE) {
      return new JsonResponse(['status' => 'error', 'message' => 'Passkey login is disabled.'], 403);
    }

    $valid1 = $this->csrfToken->validate($token, $path);
    $valid2 = $this->csrfToken->validate($token, '/' . $path);

    if (!$valid1 && !$valid2) {
      \Drupal::logger('wa_debug')->error('CSRF Fail: Token=@token, Path=@path, Valid1=@v1, Valid2=@v2', [
        '@token' => $token,
        '@path' => $path,
        '@v1' => $valid1 ? 'YES' : 'NO',
        '@v2' => $valid2 ? 'YES' : 'NO',
      ]);
      return new JsonResponse(['status' => 'error', 'message' => "Invalid CSRF token. Token=$token, Path=$path"], 403);
    }

    if (!$this->currentUserHasAllowedRole()) {
      return new JsonResponse(['status' => 'error', 'message' => 'Passkey registration not permitted for your role.'], 403);
    }

    $user = $this->entityTypeManager->getStorage('user')->load($this->currentUser()->id());
    $rpId = $this->requestStack->getCurrentRequest()->getHost();

    $options = $this->webAuthnService->getRegistrationOptions($user, $rpId);

    return new JsonResponse($options);
  }

  /**
   * Persists a newly registered credential.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request containing the credential response.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The verification result response.
   */
  public function registerVerify(Request $request) {
    $floodName = 'wa_register';
    // 1 hour
    $window = 3600;
    $maxAttempts = 5;
    $ipIdentifier = 'ip:' . ($request->getClientIp() ?? 'unknown');
    $userIdentifier = 'user:' . $this->currentUser()->id();

    if (!$this->flood->isAllowed($floodName, $maxAttempts, $window, $ipIdentifier)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Too many registration attempts. Please try again later.',
      ], 429);
    }
    if (!$this->flood->isAllowed($floodName, $maxAttempts, $window, $userIdentifier)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Too many registration attempts. Please try again later.',
      ], 429);
    }

    try {
      $config = \Drupal::config('wa.settings');
      if ($config->get('enable_passkey_login') === FALSE) {
        return new JsonResponse(['status' => 'error', 'message' => 'Passkey login is disabled.'], 403);
      }

      if (!$this->currentUserHasAllowedRole()) {
        return new JsonResponse([
          'status' => 'error',
          'message' => 'Passkey registration not permitted for your role.',
        ], 403);
      }

      $content = json_decode($request->getContent(), TRUE);
      $user = $this->entityTypeManager->getStorage('user')->load($this->currentUser()->id());
      if (!$user instanceof UserInterface) {
        throw new \Exception('User not found.');
      }
      $rpId = $request->getHost();
      $challenge = $request->getSession()->get('wa_challenge');

      if (empty($challenge)) {
        throw new \Exception('No challenge found in session.');
      }

      // Reconstruct creation options for validation.
      $publicKeyCredentialCreationOptions = PublicKeyCredentialCreationOptions::create(
        PublicKeyCredentialRpEntity::create('Web Authentication', $rpId),
        PublicKeyCredentialUserEntity::create(
          $user->getAccountName(),
          (string) $user->id(),
          $user->getDisplayName()
        ),
        Base64UrlSafe::decodeNoPadding($challenge),
        [
      // ES256.
          PublicKeyCredentialParameters::createPk(-7),
      // RS256.
          PublicKeyCredentialParameters::createPk(-257),
        ],
        NULL,
        PublicKeyCredentialCreationOptions::ATTESTATION_CONVEYANCE_PREFERENCE_NONE,
        [],
        60000
      );

      $publicKeyCredentialSource = $this->webAuthnService->validateRegistration(
        $content,
        $publicKeyCredentialCreationOptions,
        $request->getHost()
      );

      $aaguid = strtolower($publicKeyCredentialSource->aaguid->toRfc4122());
      if (!$this->passkeyLabeler->isAaguidAllowed($aaguid)) {
        $this->loggerFactory->get('wa')->warning('Passkey registration blocked for disallowed AAGUID: @aaguid', ['@aaguid' => $aaguid]);
        return new JsonResponse([
          'status' => 'error',
          'message' => 'This authenticator is not allowed for passkey login.',
        ], 403);
      }

      $credentialHash = hash('sha256', $publicKeyCredentialSource->publicKeyCredentialId);

      // Check for duplicate credential ID.
      $exists = $this->database->select('wa', 'c')
        ->fields('c', ['id'])
        ->condition('credential_hash', $credentialHash)
        ->range(0, 1)
        ->execute()
        ->fetchField();

      if ($exists) {
        throw new \Exception('This passkey is already registered.');
      }

      // Store in DB.
      $timestamp = $this->time->getRequestTime();
      $this->database->insert('wa')
        ->fields([
          'uid' => $this->currentUser()->id(),
          'credential_id' => $publicKeyCredentialSource->publicKeyCredentialId,
      // Store as base64 for safe storage.
          'public_key' => base64_encode($publicKeyCredentialSource->credentialPublicKey),
          'credential_hash' => $credentialHash,
          'user_handle' => $publicKeyCredentialSource->userHandle,
          'sign_counter' => $publicKeyCredentialSource->counter,
          'transports' => json_encode($publicKeyCredentialSource->transports),
          'aaguid' => $publicKeyCredentialSource->aaguid->toRfc4122(),
          'created' => $timestamp,
          'last_used' => $timestamp,
        ])
        ->execute();

      $this->loggerFactory->get('wa')->info('Passkey added for user @uid. Credential ID: @credential_id, AAGUID: @aaguid.', [
        '@uid' => $this->currentUser()->id(),
        '@credential_id' => base64_encode($publicKeyCredentialSource->publicKeyCredentialId),
        '@aaguid' => $publicKeyCredentialSource->aaguid->toRfc4122(),
      ]);

      return new JsonResponse(['status' => 'ok']);
    }
    catch (\Throwable $e) {
      $this->flood->register($floodName, $window, $ipIdentifier);
      $this->flood->register($floodName, $window, $userIdentifier);

      $this->loggerFactory->get('wa')->error('Passkey registration failed: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse(['status' => 'error', 'message' => 'Passkey registration failed.'], 400);
    }
    finally {
      $request->getSession()->remove('wa_challenge');
    }
  }

  /**
   * Deletes a stored passkey for the current user (or admin).
   */
  public function deletePasskey(Request $request) {
    $token = $request->headers->get('X-CSRF-Token');
    if (!$this->csrfToken->validate($token, 'wa/passkey/delete')) {
      return new JsonResponse([
        'status' => 'error',
        'message' => "Invalid CSRF token. Token=$token, Path=wa/passkey/delete",
      ], 403);
    }

    $data = json_decode($request->getContent(), TRUE) ?? [];
    $credentialId = $data['id'] ?? $request->request->get('id');

    if (empty($credentialId)) {
      return new JsonResponse(['status' => 'error', 'message' => 'Missing credential id'], 400);
    }

    $canAdminister = $this->currentUser()->hasPermission('administer all user passkey');
    if (!$canAdminister) {
      // Check if user has allowed role for API endpoint consistency.
      if (!$this->currentUserHasAllowedRole()) {
        return new JsonResponse(['status' => 'error', 'message' => 'Deletion not permitted.'], 403);
      }
    }

    $current_uid = (int) $this->currentUser()->id();

    // Accept base64-encoded IDs from the UI and raw IDs from other callers.
    $candidateIds = [$credentialId];
    $decodedBase64 = base64_decode($credentialId, TRUE);
    if ($decodedBase64 !== FALSE && $decodedBase64 !== '') {
      $candidateIds[] = $decodedBase64;
    }
    try {
      $decodedUrl = Base64UrlSafe::decodeNoPadding($credentialId);
      if ($decodedUrl !== '') {
        $candidateIds[] = $decodedUrl;
      }
    }
    catch (\Throwable $e) {
      // Ignore decode failures and continue with other candidates.
    }
    $candidateIds = array_values(array_unique(array_filter($candidateIds, static function ($value) {
      return $value !== '';
    })));

    $database = $this->database;

    // Fetch record details before deletion so we can log the owner correctly.
    $query = $database->select('wa', 'c')
      ->fields('c', ['aaguid', 'uid'])
      ->condition('credential_id', $candidateIds, 'IN');
    $result = $query->execute()->fetchAssoc();
    $target_uid = $result ? (int) $result['uid'] : NULL;
    $aaguid = $result ? $result['aaguid'] : 'Unknown';

    // Delete for owner.
    $deleted = $this->deleteByCredentialIds($candidateIds, $current_uid);

    // Allow administrators with permission to remove any credential.
    if (!$deleted && $this->currentUser()->hasPermission('administer all user passkey')) {
      $deleted = $this->deleteByCredentialIds($candidateIds);
    }

    if ($deleted) {
      $this->loggerFactory->get('wa')->info('Passkey deleted for user @uid. Credential ID: @credential_id, AAGUID: @aaguid.', [
        '@credential_id' => base64_encode($credentialId),
        '@uid' => $target_uid ?? $current_uid,
        '@aaguid' => $aaguid,
      ]);
      return new JsonResponse(['status' => 'ok']);
    }

    return new JsonResponse(['status' => 'error', 'message' => 'Deletion failed or unauthorized'], 403);
  }

  /**
   * Deletes credentials matching any of the provided IDs.
   */
  protected function deleteByCredentialIds(array $candidateIds, ?int $uid = NULL): int {
    if (empty($candidateIds)) {
      return 0;
    }

    $query = $this->database->delete('wa')
      ->condition('credential_id', $candidateIds, 'IN');

    if ($uid !== NULL) {
      $query->condition('uid', $uid);
    }

    return (int) $query->execute();
  }

  /**
   * Provides login options for passkey authentication.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The login options response.
   */
  public function loginOptions(Request $request) {
    // Check the Origin/Referer header.
    $config = \Drupal::config('wa.settings');
    if ($config->get('enable_passkey_login') === FALSE) {
      return new JsonResponse(['status' => 'error', 'message' => 'Passkey login is disabled.'], 403);
    }

    $origin = $request->headers->get('Origin');
    $referer = $request->headers->get('Referer');
    $host = $request->getHost();

    // Allow if Origin matches host (standard for CORS/fetch).
    // If Origin is missing, check Referer.
    $valid_origin = FALSE;
    if ($origin) {
      $parsed_origin = parse_url($origin);
      if (isset($parsed_origin['host']) && $parsed_origin['host'] === $host) {
        $valid_origin = TRUE;
      }
    }
    elseif ($referer) {
      $parsed_referer = parse_url($referer);
      if (isset($parsed_referer['host']) && $parsed_referer['host'] === $host) {
        $valid_origin = TRUE;
      }
    }

    if (!$valid_origin) {
      return new JsonResponse(['status' => 'error', 'message' => 'Invalid Origin or Referer.'], 403);
    }

    $rpId = $request->getHost();
    $challenge = random_bytes(32);
    $challengeEncoded = Base64UrlSafe::encodeUnpadded($challenge);
    $request->getSession()->set('wa_challenge', $challengeEncoded);

    $options = [
      'challenge' => $challengeEncoded,
      'timeout' => 60000,
      'rpId' => $rpId,
      'userVerification' => 'preferred',
    ];

    return new JsonResponse($options);
  }

  /**
   * Validates a passkey assertion and logs the user in.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request containing the assertion response.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The verification result response.
   */
  public function loginVerify(Request $request) {
    $floodName = 'wa_login';
    $window = 900;
    $maxAttempts = 5;
    $ipIdentifier = 'ip:' . ($request->getClientIp() ?? 'unknown');
    $credentialIdentifier = NULL;
    $userIdentifier = NULL;
    $registerFloodOnFailure = TRUE;

    if (!$this->flood->isAllowed($floodName, $maxAttempts, $window, $ipIdentifier)) {
      return new JsonResponse(['status' => 'error', 'message' => 'Too many login attempts. Please try again later.'], 429);
    }

    try {
      $config = \Drupal::config('wa.settings');
      if ($config->get('enable_passkey_login') === FALSE) {
        return new JsonResponse(['status' => 'error', 'message' => 'Passkey login is disabled.'], 403);
      }

      $content = json_decode($request->getContent(), TRUE);
      $credentialId = $content['id'] ?? '';
      if ($credentialId === '') {
        throw new \Exception('Missing credential id.');
      }

      $credentialIdentifier = 'cred:' . hash('sha256', $credentialId);
      if (!$this->flood->isAllowed($floodName, $maxAttempts, $window, $credentialIdentifier)) {
        return new JsonResponse(['status' => 'error', 'message' => 'Too many login attempts. Please try again later.'], 429);
      }

      $candidateIds = [$credentialId];
      $decodedBase64 = base64_decode($credentialId, TRUE);
      if ($decodedBase64 !== FALSE && $decodedBase64 !== '') {
        $candidateIds[] = $decodedBase64;
      }
      try {
        $decodedUrl = Base64UrlSafe::decodeNoPadding($credentialId);
        if ($decodedUrl !== '') {
          $candidateIds[] = $decodedUrl;
        }
      }
      catch (\Throwable $e) {
        // Ignore decode failures; we'll fall back to the raw value.
      }
      $candidateIds = array_values(array_unique(array_filter($candidateIds, static function ($value) {
        return $value !== '';
      })));

      $challenge = $request->getSession()->get('wa_challenge');

      if (empty($challenge)) {
        throw new \Exception('No challenge found in session.');
      }

      // Find user by credential ID.
      $database = $this->database;
      $query = $database->select('wa', 'c')
        ->fields('c', ['uid', 'credential_id', 'public_key', 'user_handle', 'sign_counter', 'transports', 'aaguid'])
        ->condition('credential_id', $candidateIds, 'IN')
        ->execute();

      $result = $query->fetchAssoc();

      if (!$result) {
        throw new \Exception('Credential not found.');
      }

      $uid = (int) $result['uid'];
      $user = $this->entityTypeManager->getStorage('user')->load($uid);
      $userIdentifier = 'user:' . $uid;

      if (!$user instanceof UserInterface) {
        throw new \Exception('User not found or invalid.');
      }

      if (!$user->isActive()) {
        throw new \Exception('User account is blocked.');
      }

      // Check if the user is allowed to login with passkey.
      $allowed_roles = $config->get('allowed_roles') ?? [];
      $allowed_roles = array_filter($allowed_roles);
      if (!empty($allowed_roles)) {
        $user_roles = $user->getRoles();
        if (empty(array_intersect($user_roles, $allowed_roles))) {
          throw new \Exception('Passkey login not permitted for this account role.');
        }
      }

      if (!$this->passkeyLabeler->isAaguidAllowed(strtolower((string) $result['aaguid']))) {
        throw new \Exception('This authenticator is not allowed for passkey login.');
      }

      // Reconstruct PublicKeyCredentialSource.
      $storedCredentialId = $result['credential_id'];
      $storedPublicKey = $result['public_key'] ?? '';
      // Prefer base64-decoded value;
      // fall back to JSON-decoded legacy format or raw.
      $decodedPublicKey = base64_decode($storedPublicKey, TRUE);
      if ($decodedPublicKey === FALSE) {
        $jsonDecoded = json_decode($storedPublicKey, TRUE);
        if (is_string($jsonDecoded)) {
          $decodedPublicKey = $jsonDecoded;
        }
        else {
          $decodedPublicKey = (string) $storedPublicKey;
        }
      }

      $publicKeyCredentialSource = PublicKeyCredentialSource::create(
        $storedCredentialId,
        PublicKeyCredentialDescriptor::CREDENTIAL_TYPE_PUBLIC_KEY,
        json_decode($result['transports'] ?? '[]', TRUE),
      // Assumed attestation type.
        'none',
        new EmptyTrustPath(),
        Uuid::fromString($result['aaguid']),
        $decodedPublicKey,
        $result['user_handle'],
        (int) $result['sign_counter']
      );

      // Reconstruct options.
      $challengeBinary = Base64UrlSafe::decodeNoPadding($challenge);
      $rpId = $request->getHost();
      $allowCredentials = [
        PublicKeyCredentialDescriptor::create(
          PublicKeyCredentialDescriptor::CREDENTIAL_TYPE_PUBLIC_KEY,
          $storedCredentialId,
          json_decode($result['transports'] ?? '[]', TRUE)
        ),
      ];
      $userVerification = PublicKeyCredentialRequestOptions::USER_VERIFICATION_REQUIREMENT_PREFERRED;
      $timeout = 60000;

      $publicKeyCredentialRequestOptions = PublicKeyCredentialRequestOptions::create(
        $challengeBinary,
        $rpId,
        $allowCredentials,
        $userVerification,
        $timeout
      );

      $publicKeyCredentialSource = $this->webAuthnService->validateAssertion(
        $content,
        $publicKeyCredentialRequestOptions,
        $publicKeyCredentialSource,
        $request->getHost(),
        $result['user_handle']
      );

      // Update counter and last used.
      $database->update('wa')
        ->fields([
          'last_used' => $this->time->getRequestTime(),
          'sign_counter' => $publicKeyCredentialSource->counter,
        ])
        ->condition('credential_id', $storedCredentialId)
        ->execute();

      user_login_finalize($user);
      $this->loggerFactory->get('wa')->info('User logged in with passkey. User: @uid, Credential ID: @credential_id.', [
        '@uid' => $uid,
        '@credential_id' => $credentialId,
      ]);

      $registerFloodOnFailure = FALSE;
      return new JsonResponse(['status' => 'ok', 'uid' => $uid]);
    }
    catch (\Throwable $e) {
      if ($registerFloodOnFailure) {
        $this->flood->register($floodName, $window, $ipIdentifier);
        if ($credentialIdentifier) {
          $this->flood->register($floodName, $window, $credentialIdentifier);
        }
        if ($userIdentifier) {
          $this->flood->register($floodName, $window, $userIdentifier);
        }
      }

      $this->loggerFactory->get('wa')->error('Passkey login failed: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse(['status' => 'error', 'message' => 'Passkey login failed.'], 400);
    }
    finally {
      // Clear the challenge to enforce one-time use.
      $request->getSession()->remove('wa_challenge');
    }
  }

}
