<?php

namespace Drupal\wa\Service;

use Cose\Algorithm\Manager;
use Cose\Algorithm\Signature\ECDSA\ES256;
use Cose\Algorithm\Signature\RSA\RS256;
use ParagonIE\ConstantTime\Base64UrlSafe;
use Webauthn\AttestationStatement\AttestationObjectLoader;
use Webauthn\AttestationStatement\AttestationStatementSupportManager;
use Webauthn\AttestationStatement\NoneAttestationStatementSupport;
use Webauthn\AuthenticatorAssertionResponse;
use Webauthn\AuthenticatorAssertionResponseValidator;
use Webauthn\AuthenticatorAttestationResponse;
use Webauthn\AuthenticatorAttestationResponseValidator;
use Webauthn\AuthenticatorDataLoader;
use Webauthn\CeremonyStep\CeremonyStepManagerFactory;
use Webauthn\CollectedClientData;
use Webauthn\PublicKeyCredentialCreationOptions;
use Webauthn\PublicKeyCredentialRequestOptions;
use Webauthn\PublicKeyCredentialSource;

/**
 * Service for WebAuthn operations.
 */
class WebAuthnService {

  /**
   * Creates a configured CeremonyStepManagerFactory.
   */
  public function getValidatorFactory(): CeremonyStepManagerFactory {
    $factory = new CeremonyStepManagerFactory();
    $factory->setAlgorithmManager(Manager::create()->add(ES256::create(), RS256::create()));
    $factory->setAttestationStatementSupportManager(new AttestationStatementSupportManager([
      new NoneAttestationStatementSupport(),
    ]));
    return $factory;
  }

  /**
   * Decodes a Base64 URL-encoded string.
   */
  public function base64UrlDecode(string $data): ?string {
    try {
      return Base64UrlSafe::decodeNoPadding($data);
    }
    catch (\Throwable $e) {
      return NULL;
    }
  }

  /**
   * Validates a registration response.
   */
  public function validateRegistration(array $content, PublicKeyCredentialCreationOptions $options, string $host): PublicKeyCredentialSource {
    // Load the response.
    $attestationStatementSupportManager = new AttestationStatementSupportManager([
      new NoneAttestationStatementSupport(),
    ]);
    $attestationObjectLoader = AttestationObjectLoader::create($attestationStatementSupportManager);
    $clientDataJSON = CollectedClientData::createFormJson($content['response']['clientDataJSON']);
    $attestationObject = $attestationObjectLoader->load($content['response']['attestationObject']);

    $authenticatorAttestationResponse = AuthenticatorAttestationResponse::create(
      $clientDataJSON,
      $attestationObject,
      $content['response']['transports'] ?? []
    );

    // Validate.
    $validator = AuthenticatorAttestationResponseValidator::create(
      $this->getValidatorFactory()->creationCeremony()
    );

    return $validator->check(
      $authenticatorAttestationResponse,
      $options,
      $host
    );
  }

  /**
   * Validates a login assertion.
   */
  public function validateAssertion(array $content, PublicKeyCredentialRequestOptions $options, PublicKeyCredentialSource $source, string $host, ?string $userHandle): PublicKeyCredentialSource {
    // Create response object.
    $clientDataJSON = CollectedClientData::createFormJson($content['response']['clientDataJSON']);
    $authenticatorData = AuthenticatorDataLoader::create()->load(
      Base64UrlSafe::decodeNoPadding($content['response']['authenticatorData'])
    );
    $authenticatorAssertionResponse = AuthenticatorAssertionResponse::create(
      $clientDataJSON,
      $authenticatorData,
      Base64UrlSafe::decodeNoPadding($content['response']['signature']),
      Base64UrlSafe::decodeNoPadding($content['response']['userHandle'] ?? '')
    );

    // Validate.
    $validator = AuthenticatorAssertionResponseValidator::create(
      $this->getValidatorFactory()->requestCeremony()
    );

    return $validator->check(
      $source,
      $authenticatorAssertionResponse,
      $options,
      $host,
      $userHandle
    );
  }

  /**
   * Generates registration options.
   */
  public function getRegistrationOptions($user, string $rpId): array {
    // Generate random challenge.
    $challenge = random_bytes(32);
    $challengeEncoded = Base64UrlSafe::encodeUnpadded($challenge);
    \Drupal::request()->getSession()->set('wa_challenge', $challengeEncoded);

    // Simplified options generation.
    return [
      'rp' => [
        'name' => 'Web Authentication',
        'id' => $rpId,
      ],
      'user' => [
        'id' => Base64UrlSafe::encodeUnpadded((string) $user->id()),
        'name' => $user->getAccountName(),
        'displayName' => $user->getDisplayName(),
      ],
      'challenge' => $challengeEncoded,
      'pubKeyCredParams' => [
          // ES256.
              ['type' => 'public-key', 'alg' => -7],
      // RS256.
              ['type' => 'public-key', 'alg' => -257],
      ],
      'timeout' => 60000,
      'attestation' => 'none',
    ];
  }

}
