<?php

namespace Drupal\Tests\wa\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests security aspects of WebAuthn login and registration.
 *
 * @group wa
 */
class PasskeySecurityTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['wa'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests unauthorized registration attempts.
   */
  public function testUnauthorizedRegistration() {
    // Create a user without the allowed role.
    $noRoleUser = $this->drupalCreateUser([]);
    $this->drupalLogin($noRoleUser);

    // Restrict passkeys to a specific role.
    $config = \Drupal::configFactory()->getEditable('wa.settings');
    $config->set('allowed_roles', ['administrator'])->save();

    $token = \Drupal::service('csrf_token')->get('wa/register/options');

    // Attempt to access registration options.
    $this->drupalPostJson('/wa/register/options', [], [], ['X-CSRF-Token' => $token]);
    $this->assertSession()->statusCodeEquals(403);

    // Attempt to submit registration verification.
    $this->drupalPostJson('/wa/register/verify', []);
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests registration with missing or invalid challenge.
   */
  public function testRegistrationChallengeValidation() {
    $user = $this->drupalCreateUser([]);
    $this->drupalLogin($user);

    // Get token from the management page.
    $this->drupalGet('/user/' . $user->id() . '/passkeys');
    $settings = $this->getDrupalSettings();

    $registerToken = $settings['wa']['registerOptionsToken'] ?? '';

    // Case 1: Missing challenge in session.
    // We send a request without first calling
    // (which sets the challenge).
    $this->drupalPostJson('/wa/register/verify', ['some' => 'data'], [], ['X-CSRF-Token' => $registerToken]);
    // The controller throws an exception
    // "No challenge found in session", caught and returns 400.
    $this->assertSession()->statusCodeEquals(400);
    $this->assertSession()->responseContains('Passkey registration failed.');

    // Case 2: Invalid challenge.
    $this->drupalPostJson('/wa/register/options', [], [], ['X-CSRF-Token' => $registerToken]);

    $this->assertSession()->statusCodeEquals(200);

    // Now send garbage data.
    // The WebAuthn library will fail to validate it against the challenge.
    $this->drupalPostJson('/wa/register/verify', [
      'id' => 'garbage',
      'rawId' => 'garbage',
      'response' => ['clientDataJSON' => 'garbage'],
    ], [], ['X-CSRF-Token' => $registerToken]);
    $this->assertSession()->statusCodeEquals(400);
  }

  /**
   * Tests login with missing or invalid challenge.
   */
  public function testLoginChallengeValidation() {
    $user = $this->drupalCreateUser([]);
    // Create a dummy credential in DB to pass the initial lookup.
    $credentialId = 'test-cred-id';
    $encodedId = base64_encode($credentialId);

    \Drupal::database()->insert('wa')
      ->fields([
        'uid' => $user->id(),
        'credential_id' => $credentialId,
        'credential_hash' => hash('sha256', $credentialId),
        'public_key' => base64_encode('publickey'),
        'user_handle' => 'handle',
        'sign_counter' => 0,
        'transports' => json_encode([]),
        'aaguid' => 'fbfc3007-154e-4ecc-8c0b-6e020557d7bd',
        'created' => \Drupal::time()->getRequestTime(),
        'last_used' => \Drupal::time()->getRequestTime(),
      ])
      ->execute();

    // Get token from login page.
    if (\Drupal::currentUser()->isAuthenticated()) {
      $this->drupalLogout();
    }
    $this->drupalGet('/user/login');
    $settings = $this->getDrupalSettings();
    $loginToken = $settings['wa']['loginOptionsToken'] ?? '';

    // Case 1: Missing challenge.
    // Login options now requires Origin header, not CSRF token.
    // We simulate a valid options request first to set the challenge.
    $this->drupalPostJson('/wa/login/options', [], [], ['Origin' => $this->baseUrl]);

    // Test "Missing challenge",.
    $this->getSession()->restart();
    $this->drupalPostJson('/wa/login/verify', ['id' => $encodedId], [], ['X-CSRF-Token' => $loginToken]);
    $this->assertSession()->statusCodeEquals(400);
    $this->assertSession()->responseContains('Passkey login failed.');

    // Case 2: Challenge exists but verification fails (garbage data).
    $this->drupalPostJson('/wa/login/options', [], [], ['Origin' => $this->baseUrl]);
    // Ensure this succeeds.
    $this->assertSession()->statusCodeEquals(200);
    $this->drupalPostJson('/wa/login/verify', [
      'id' => $encodedId,
      'rawId' => $encodedId,
      'response' => ['clientDataJSON' => 'garbage'],
    ], [], ['X-CSRF-Token' => $loginToken]);
    $this->assertSession()->statusCodeEquals(400);
  }

  /**
   * Tests login attempts with non-existent credentials.
   */
  public function testNonExistentCredential() {
    $this->drupalGet('/user/login');
    $settings = $this->getDrupalSettings();
    $loginToken = $settings['wa']['loginOptionsToken'] ?? '';

    // Use Origin header instead of CSRF token.
    $this->drupalPostJson('/wa/login/options', [], [], ['Origin' => $this->baseUrl]);

    // Ensure challenge is set.
    $this->assertSession()->statusCodeEquals(200);

    $this->drupalPostJson('/wa/login/verify', ['id' => base64_encode('non-existent')], [], ['X-CSRF-Token' => $loginToken]);
    $this->assertSession()->statusCodeEquals(400);
    // Should fail with a generic message.
    $this->assertSession()->responseContains('Passkey login failed.');
  }

  /**
   * Tests login attempts by unauthorized users.
   */
  public function testUnauthorizedUserLogin() {
    // Restrict passkeys to a specific role.
    $config = \Drupal::configFactory()->getEditable('wa.settings');
    $config->set('allowed_roles', ['administrator'])->save();

    // User without the allowed role.
    $user = $this->drupalCreateUser([]);
    $credentialId = 'test-cred-no-perm';
    $encodedId = base64_encode($credentialId);

    \Drupal::database()->insert('wa')
      ->fields([
        'uid' => $user->id(),
        'credential_id' => $credentialId,
        'credential_hash' => hash('sha256', $credentialId),
        'public_key' => base64_encode('publickey'),
        'user_handle' => 'handle',
        'sign_counter' => 0,
        'transports' => json_encode([]),
        'aaguid' => 'fbfc3007-154e-4ecc-8c0b-6e020557d7bd',
        'created' => \Drupal::time()->getRequestTime(),
        'last_used' => \Drupal::time()->getRequestTime(),
      ])
      ->execute();

    $loginToken = \Drupal::service('csrf_token')->get('wa/login/options');
    $this->drupalPostJson('/wa/login/options', [], [], ['Origin' => $this->baseUrl]);
    $this->drupalPostJson('/wa/login/verify', ['id' => $encodedId]);

    // Should fail because user lacks role.
    $this->assertSession()->statusCodeEquals(400);
    $this->assertSession()->responseContains('Passkey login failed.');

    // Blocked users cannot log in even with valid credentials.
    // Reset allowed roles so this user can try.
    $config->set('allowed_roles', [])->save();
    $blockedUser = $this->drupalCreateUser([]);
    $blockedUser->block();
    $blockedUser->save();
    $blockedCredentialId = 'blocked-cred-id';
    $blockedEncodedId = base64_encode($blockedCredentialId);

    \Drupal::database()->insert('wa')
      ->fields([
        'uid' => $blockedUser->id(),
        'credential_id' => $blockedCredentialId,
        'credential_hash' => hash('sha256', $blockedCredentialId),
        'public_key' => base64_encode('publickey'),
        'user_handle' => 'handle',
        'sign_counter' => 0,
        'transports' => json_encode([]),
        'aaguid' => 'fbfc3007-154e-4ecc-8c0b-6e020557d7bd',
        'created' => \Drupal::time()->getRequestTime(),
        'last_used' => \Drupal::time()->getRequestTime(),
      ])
      ->execute();

    $blockedLoginToken = $this->getSessionCsrfToken();
    $this->drupalPostJson('/wa/login/options', [], [], ['Origin' => $this->baseUrl]);
    $this->drupalPostJson('/wa/login/verify', ['id' => $blockedEncodedId]);
    $this->assertSession()->statusCodeEquals(400);
    $this->assertSession()->responseContains('Passkey login failed.');
  }

  /**
   * Performs a JSON POST request.
   *
   * @param string $path
   *   The path to request.
   * @param array $data
   *   The JSON data to send.
   * @param array $options
   *   (optional) Options to be passed to buildUrl().
   * @param array $headers
   *   (optional) Headers to send with the request.
   */
  protected function drupalPostJson($path, array $data, array $options = [], array $headers = []) {
    $content = json_encode($data);
    $server = ['CONTENT_TYPE' => 'application/json'];
    foreach ($headers as $name => $value) {
      $server['HTTP_' . strtoupper(str_replace('-', '_', $name))] = $value;
    }
    $this->getSession()->getDriver()->getClient()->request(
      'POST',
      $this->buildUrl($path, $options),
      [],
      [],
      $server,
      $content
    );
  }

  /**
   * Returns a CSRF token for the current session.
   */
  protected function getSessionCsrfToken(): string {
    $this->drupalGet('/session/token');
    return trim($this->getSession()->getPage()->getContent());
  }

}
