(function ($, Drupal, drupalSettings) {

  'use strict';

  // Helper to decode Base64URL to ArrayBuffer
  function base64UrlToBuffer(base64url) {
    const padding = '='.repeat((4 - base64url.length % 4) % 4);
    const base64 = (base64url + padding).replace(/-/g, '+').replace(/_/g, '/');
    const rawData = window.atob(base64);
    const outputArray = new Uint8Array(rawData.length);
    for (let i = 0; i < rawData.length; ++i) {
      outputArray[i] = rawData.charCodeAt(i);
    }
    return outputArray.buffer;
  }

  // Helper to encode ArrayBuffer to Base64URL
  function bufferToBase64Url(buffer) {
    const bytes = new Uint8Array(buffer);
    let binary = '';
    for (let i = 0; i < bytes.byteLength; i++) {
      binary += String.fromCharCode(bytes[i]);
    }
    return window.btoa(binary)
      .replace(/\+/g, '-')
      .replace(/\//g, '_')
      .replace(/=/g, '');
  }

  function getCsrfToken() {
    const settingsToken = drupalSettings.wa && drupalSettings.wa.passkeyDeleteToken;

    if (settingsToken) {
      return Promise.resolve(settingsToken);
    }

    return Promise.reject(new Error('Missing passkey CSRF token'));
  }

  function getOptionsToken(key) {
    const settingsToken = drupalSettings.wa && drupalSettings.wa[key];
    if (settingsToken) {
      return Promise.resolve(settingsToken);
    }
    return Promise.reject(new Error('Missing passkey options CSRF token'));
  }

  function getRoute(key) {
    const route = drupalSettings.wa && drupalSettings.wa.routes && drupalSettings.wa.routes[key];
    if (route) {
      return route;
    }
    // Fallback for pages where routes might not be populated (though they should be if the library is loaded)
    // or if the user hasn't updated the module code yet.
    console.warn('Missing route definition for ' + key + ', falling back to default.');
    const defaults = {
      'loginOptions': '/wa/login/options',
      'loginVerify': '/wa/login/verify',
      'registerOptions': '/wa/register/options',
      'registerVerify': '/wa/register/verify',
      'passkeyDelete': '/wa/passkey/delete'
    };
    return defaults[key];
  }

  Drupal.behaviors.waPasskey = {
    attach: function (context, settings) {
      // Registration
      const registerBtn = once('wa-register', '#register-passkey', context);
      registerBtn.forEach(element => {
        element.addEventListener('click', function (e) {
          e.preventDefault();

          getOptionsToken('registerOptionsToken')
            .then(csrfToken => {
              return fetch(getRoute('registerOptions'), {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                  'X-CSRF-Token': csrfToken
                },
              });
            })
            .then(response => response.json())
            .then(options => {
              options.challenge = base64UrlToBuffer(options.challenge);
              options.user.id = base64UrlToBuffer(options.user.id);

              return navigator.credentials.create({ publicKey: options });
            })
            .then(credential => {
              const transports = typeof credential.response.getTransports === 'function'
                ? credential.response.getTransports()
                : [];
              const response = {
                id: credential.id,
                rawId: bufferToBase64Url(credential.rawId),
                type: credential.type,
                response: {
                  attestationObject: bufferToBase64Url(credential.response.attestationObject),
                  clientDataJSON: bufferToBase64Url(credential.response.clientDataJSON),
                  transports: transports,
                },
              };

              return getOptionsToken('registerVerifyToken')
                .then(csrfToken => fetch(getRoute('registerVerify'), {
                  method: 'POST',
                  headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-Token': csrfToken
                  },
                  body: JSON.stringify(response)
                }));
            })
            .then(response => response.json())
            .then(data => {
              if (data.status === 'ok') {
                alert('Passkey registered successfully!');
                location.reload();
              } else {
                alert('Registration failed.');
              }
            })
            .catch(err => {
              console.error(err);
              alert('Error registering passkey: ' + err.message);
            });
        });
      });

      // Delete passkey
      const deleteButtons = once('wa-delete', '.passkey-delete', context);
      deleteButtons.forEach(button => {
        button.addEventListener('click', function (e) {
          e.preventDefault();
          const credentialId = button.dataset.credentialId;
          if (!credentialId) {
            return;
          }

          const confirmed = window.confirm('Are you sure you want to delete this passkey? This action cannot be undone.');
          if (!confirmed) {
            return;
          }

          getCsrfToken()
            .then(csrfToken => {
              const deleteUrl = getRoute('passkeyDelete') || '/wa/passkey/delete';

              return fetch(deleteUrl, {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                  'Accept': 'application/json',
                  'X-CSRF-Token': csrfToken
                },
                body: JSON.stringify({ id: credentialId }),
                credentials: 'include'
              });
            })
            .then(response => response.json())
            .then(data => {
              if (data.status === 'ok') {
                location.reload();
              }
              else {
                alert(data.message || 'Delete failed.');
              }
            })
            .catch(err => {
              console.error(err);
              alert('Error deleting passkey: ' + err.message);
            });
        });
      });

      // Login
      const loginBtn = once('wa-login', '#wa-login-btn', context);
      loginBtn.forEach(element => {
        element.addEventListener('click', function (e) {
          e.preventDefault();

          fetch(getRoute('loginOptions'), {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json'
            },
          })
            .then(response => response.json())
            .then(options => {
              options.challenge = base64UrlToBuffer(options.challenge);

              return navigator.credentials.get({ publicKey: options });
            })
            .then(credential => {
              const response = {
                id: credential.id,
                rawId: bufferToBase64Url(credential.rawId),
                type: credential.type,
                response: {
                  authenticatorData: bufferToBase64Url(credential.response.authenticatorData),
                  clientDataJSON: bufferToBase64Url(credential.response.clientDataJSON),
                  signature: bufferToBase64Url(credential.response.signature),
                  userHandle: credential.response.userHandle ? bufferToBase64Url(credential.response.userHandle) : null,
                },
              };

              return fetch(getRoute('loginVerify'), {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(response)
              });
            })
            .then(response => response.json())
            .then(data => {
              if (data.status === 'ok') {
                window.location.href = '/user/' + data.uid;
              } else {
                alert('Login failed: ' + data.message);
              }
            })
            .catch(err => {
              console.error(err);
              alert('Error logging in: ' + err.message);
            });
        });
      });
    }
  };

})(jQuery, Drupal, drupalSettings);
