<?php

namespace Drupal\wa\Form;

use Drupal\user\RoleInterface;
use Drupal\user\Entity\Role;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for WA module.
 */
class PasskeySettingsForm extends ConfigFormBase {

  /**
   * The labeler service.
   *
   * @var \Drupal\wa\Service\PasskeyLabeler
   */
  protected $passkeyLabeler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    /** @var static $instance */
    $instance = parent::create($container);
    $instance->passkeyLabeler = $container->get('wa.passkey_labeler');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'wa_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['wa.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('wa.settings');
    $allowed = $config->get('allowed_aaguids') ?? [];
    $customLabels = $config->get('custom_aaguid_labels') ?? [];
    $enablePasskeyLogin = $config->get('enable_passkey_login') ?? TRUE;
    $allowedRoles = $config->get('allowed_roles') ?? [];

    // Known provider options keyed by AAGUID.
    $providerOptions = $this->passkeyLabeler->getProviderOptions();
    $knownAaguids = array_keys($providerOptions);

    $allowedKnown = array_values(array_intersect($allowed, $knownAaguids));
    $allowedCustom = array_values(array_diff($allowed, $knownAaguids));
    $allowedCustomWithLabels = array_map(function ($aaguid) use ($customLabels) {
      $label = $customLabels[$aaguid] ?? '';
      return $label !== '' ? $aaguid . '|' . $label : $aaguid;
    }, $allowedCustom);

    $form['enable_passkey_login'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Passkey Login'),
      '#default_value' => $enablePasskeyLogin,
      '#description' => $this->t('If unchecked, the "Sign in with Passkey" button will be hidden and login/registration endpoints will be disabled.'),
    ];

    $form['user_verification'] = [
      '#type' => 'radios',
      '#title' => $this->t('User Verification Requirement'),
      '#default_value' => $config->get('user_verification') ?? 'preferred',
      '#options' => [
        'preferred' => $this->t('Preferred (Allows user presence only)'),
        'required' => $this->t('Required (Enforces PIN/Biometrics)'),
      ],
      '#description' => $this->t('Choose "Required" for higher security, or "Preferred" for better compatibility.'),
      '#required' => TRUE,
    ];

    $roles = Role::loadMultiple();
    $roleOptions = [];
    foreach ($roles as $role) {
      // Filter out anonymous.
      if ($role->id() !== RoleInterface::ANONYMOUS_ID) {
        $roleOptions[$role->id()] = $role->label();
      }
    }

    $form['allowed_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed Roles'),
      '#options' => $roleOptions,
      '#default_value' => $allowedRoles,
      '#description' => $this->t('Select which user roles are allowed to use passkeys. If no roles are selected, no one can use passkeys.'),
    ];

    $form['allowed_known'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed authenticators'),
      '#options' => $providerOptions,
      '#default_value' => $allowedKnown,
      '#description' => $this->t('Select which known authenticators (by AAGUID) you allow for registration and login.'),
    ];

    $form['allowed_custom'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Additional AAGUIDs'),
      '#default_value' => implode("\n", $allowedCustomWithLabels),
      '#description' => $this->t('Enter one AAGUID per line (lowercase, hex with dashes). Optionally add a label after a pipe, e.g. 12345678-1234-1234-1234-1234567890ab|My Authenticator.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $customRaw = $form_state->getValue('allowed_custom') ?? '';
    $customLines = array_map('trim', preg_split('/\r\n|\r|\n/', $customRaw));
    $customLines = array_filter($customLines, static function ($value) {
      return $value !== '';
    });

    $invalid = [];
    $parsedCustom = [];
    foreach ($customLines as $line) {
      [$aaguid, $label] = array_pad(explode('|', $line, 2), 2, '');
      $aaguid = strtolower(trim($aaguid));
      $label = trim($label);

      if (!preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/', $aaguid)) {
        $invalid[] = $line;
        continue;
      }

      $parsedCustom[] = ['aaguid' => $aaguid, 'label' => $label];
    }

    if (!empty($invalid)) {
      $form_state->setErrorByName('allowed_custom', $this->t('Invalid AAGUID(s): @list', ['@list' => implode(', ', $invalid)]));
    }
    else {
      $form_state->setValue('parsed_custom', $parsedCustom);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $allowedKnown = array_filter($form_state->getValue('allowed_known') ?? []);
    $allowedRoles = array_filter($form_state->getValue('allowed_roles') ?? []);
    $enablePasskeyLogin = $form_state->getValue('enable_passkey_login');
    $userVerification = $form_state->getValue('user_verification');

    $customRaw = $form_state->getValue('allowed_custom') ?? '';
    $parsedCustom = $form_state->getValue('parsed_custom') ?? [];

    $customAaguids = array_map(static function ($item) {
      return $item['aaguid'];
    }, $parsedCustom);

    $merged = array_merge($allowedKnown, $customAaguids);
    $merged = array_map(static function ($value) {
      return strtolower($value);
    }, $merged);
    $merged = array_values(array_unique(array_filter($merged, static function ($value) {
      return $value !== '';
    })));

    $customLabels = [];
    foreach ($parsedCustom as $pair) {
      if ($pair['label'] !== '') {
        $customLabels[$pair['aaguid']] = $pair['label'];
      }
    }

    $this->configFactory->getEditable('wa.settings')
      ->set('allowed_aaguids', $merged)
      ->set('custom_aaguid_labels', $customLabels)
      ->set('enable_passkey_login', $enablePasskeyLogin)
      ->set('user_verification', $userVerification)
      ->set('allowed_roles', $allowedRoles)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
