<?php

namespace Drupal\wa\Service;

use Cose\Algorithm\Manager;
use Cose\Algorithm\Signature\ECDSA\ES256;
use Cose\Algorithm\Signature\RSA\RS256;
use ParagonIE\ConstantTime\Base64UrlSafe;
use Webauthn\AttestationStatement\AttestationObjectLoader;
use Webauthn\AttestationStatement\AttestationStatementSupportManager;
use Webauthn\AttestationStatement\NoneAttestationStatementSupport;
use Webauthn\AuthenticatorAssertionResponse;
use Webauthn\AuthenticatorAssertionResponseValidator;
use Webauthn\AuthenticatorAttestationResponse;
use Webauthn\AuthenticatorAttestationResponseValidator;
use Webauthn\AuthenticatorDataLoader;
use Webauthn\CeremonyStep\CeremonyStepManagerFactory;
use Webauthn\CollectedClientData;
use Webauthn\PublicKeyCredentialDescriptor;
use Webauthn\PublicKeyCredentialCreationOptions;
use Webauthn\PublicKeyCredentialRequestOptions;
use Webauthn\PublicKeyCredentialSource;
use Webauthn\TrustPath\EmptyTrustPath;
use Symfony\Component\Uid\Uuid;
use Webauthn\PublicKeyCredentialRpEntity;
use Webauthn\PublicKeyCredentialUserEntity;
use Webauthn\PublicKeyCredentialParameters;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\user\UserInterface;

/**
 * Service for WebAuthn operations.
 */
class WebAuthnService {

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The validator factory.
   *
   * @var \Webauthn\CeremonyStep\CeremonyStepManagerFactory
   */
  protected $validatorFactory;

  /**
   * Constructs a WebAuthnService object.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(RequestStack $request_stack) {
    $this->requestStack = $request_stack;
    $this->validatorFactory = new CeremonyStepManagerFactory();
    $this->validatorFactory->setAlgorithmManager(Manager::create()->add(ES256::create(), RS256::create()));
    $this->validatorFactory->setAttestationStatementSupportManager(new AttestationStatementSupportManager([
      new NoneAttestationStatementSupport(),
    ]));
  }

  /**
   * Creates a configured CeremonyStepManagerFactory.
   */
  public function getValidatorFactory(): CeremonyStepManagerFactory {
    return $this->validatorFactory;
  }

  /**
   * Decodes a Base64 URL-encoded string.
   */
  public function base64UrlDecode(string $data): ?string {
    try {
      return Base64UrlSafe::decodeNoPadding($data);
    }
    catch (\Throwable $e) {
      return NULL;
    }
  }

  /**
   * Validates a registration response.
   */
  public function validateRegistration(array $content, PublicKeyCredentialCreationOptions $options, string $host): PublicKeyCredentialSource {
    // Load the response.
    $attestationStatementSupportManager = new AttestationStatementSupportManager([
      new NoneAttestationStatementSupport(),
    ]);
    $attestationObjectLoader = AttestationObjectLoader::create($attestationStatementSupportManager);
    $clientDataJSON = CollectedClientData::createFormJson($content['response']['clientDataJSON']);
    $attestationObject = $attestationObjectLoader->load($content['response']['attestationObject']);

    $authenticatorAttestationResponse = AuthenticatorAttestationResponse::create(
      $clientDataJSON,
      $attestationObject,
      $content['response']['transports'] ?? []
    );

    // Validate.
    $validator = AuthenticatorAttestationResponseValidator::create(
      $this->getValidatorFactory()->creationCeremony()
    );

    return $validator->check(
      $authenticatorAttestationResponse,
      $options,
      $host
    );
  }

  /**
   * Builds and validates registration options from request data.
   *
   * @param array $content
   *   The client-provided registration payload.
   * @param string $rpId
   *   The relying party ID.
   * @param string $challenge
   *   The base64url-encoded challenge.
   * @param string $host
   *   The current request host.
   * @param string $username
   *   The account name.
   * @param string $userId
   *   The user ID as string.
   * @param string $displayName
   *   The display name.
   *
   * @return \Webauthn\PublicKeyCredentialSource
   *   The validated credential source.
   */
  public function validateRegistrationRequest(array $content, string $rpId, string $challenge, string $host, string $username, string $userId, string $displayName): PublicKeyCredentialSource {
    $publicKeyCredentialCreationOptions = PublicKeyCredentialCreationOptions::create(
      PublicKeyCredentialRpEntity::create('Web Authentication', $rpId),
      PublicKeyCredentialUserEntity::create(
        $username,
        $userId,
        $displayName
      ),
      Base64UrlSafe::decodeNoPadding($challenge),
      [
        // ES256.
        PublicKeyCredentialParameters::createPk(-7),
        // RS256.
        PublicKeyCredentialParameters::createPk(-257),
      ],
      NULL,
      PublicKeyCredentialCreationOptions::ATTESTATION_CONVEYANCE_PREFERENCE_NONE,
      [],
      60000
    );

    return $this->validateRegistration(
      $content,
      $publicKeyCredentialCreationOptions,
      $host
    );
  }

  /**
   * Validates a login assertion.
   */
  public function validateAssertion(array $content, PublicKeyCredentialRequestOptions $options, PublicKeyCredentialSource $source, string $host, ?string $userHandle): PublicKeyCredentialSource {
    // Create response object.
    $clientDataJSON = CollectedClientData::createFormJson($content['response']['clientDataJSON']);
    $authenticatorData = AuthenticatorDataLoader::create()->load(
      Base64UrlSafe::decodeNoPadding($content['response']['authenticatorData'])
    );
    $authenticatorAssertionResponse = AuthenticatorAssertionResponse::create(
      $clientDataJSON,
      $authenticatorData,
      Base64UrlSafe::decodeNoPadding($content['response']['signature']),
      Base64UrlSafe::decodeNoPadding($content['response']['userHandle'] ?? '')
    );

    // Validate.
    $validator = AuthenticatorAssertionResponseValidator::create(
      $this->getValidatorFactory()->requestCeremony()
    );

    return $validator->check(
      $source,
      $authenticatorAssertionResponse,
      $options,
      $host,
      $userHandle
    );
  }

  /**
   * Generates registration options.
   */
  public function getRegistrationOptions(UserInterface $user, string $rpId): array {
    // Generate random challenge.
    $challenge = random_bytes(32);
    $challengeEncoded = Base64UrlSafe::encodeUnpadded($challenge);
    $this->requestStack->getCurrentRequest()->getSession()->set('wa_challenge', $challengeEncoded);

    // Simplified options generation.
    return [
      'rp' => [
        'name' => 'Web Authentication',
        'id' => $rpId,
      ],
      'user' => [
        'id' => Base64UrlSafe::encodeUnpadded((string) $user->id()),
        'name' => $user->getAccountName(),
        'displayName' => $user->getDisplayName(),
      ],
      'challenge' => $challengeEncoded,
      'pubKeyCredParams' => [
          // ES256.
              ['type' => 'public-key', 'alg' => -7],
      // RS256.
              ['type' => 'public-key', 'alg' => -257],
      ],
      'timeout' => 60000,
      'attestation' => 'none',
    ];
  }

  /**
   * Validates a login request using stored credential data.
   *
   * @param array $content
   *   The client-provided WebAuthn assertion payload.
   * @param array $credentialData
   *   The credential data fetched from storage (must include credential_id,
   *   transports, aaguid, public_key, user_handle, sign_counter).
   * @param string $challenge
   *   The base64url-encoded challenge from the session.
   * @param string $rpId
   *   The relying party ID (usually the request host).
   * @param string $host
   *   The current request host.
   * @param string $userVerificationRequirement
   *   The user verification requirement ('preferred' or 'required').
   * @param \Webauthn\PublicKeyCredentialSource|null $validatedSource
   *   Populated with the validated credential source on success.
   *
   * @return bool
   *   TRUE on successful validation; otherwise, an exception is thrown.
   */
  public function validateLoginRequest(array $content, array $credentialData, string $challenge, string $rpId, string $host, string $userVerificationRequirement, ?PublicKeyCredentialSource &$validatedSource = NULL): bool {
    $storedCredentialId = $credentialData['credential_id'];
    $storedPublicKey = $credentialData['public_key'] ?? '';
    $decodedPublicKey = base64_decode($storedPublicKey, TRUE);
    if ($decodedPublicKey === FALSE) {
      $jsonDecoded = json_decode($storedPublicKey, TRUE);
      if (is_string($jsonDecoded)) {
        $decodedPublicKey = $jsonDecoded;
      }
      else {
        $decodedPublicKey = (string) $storedPublicKey;
      }
    }

    $publicKeyCredentialSource = PublicKeyCredentialSource::create(
      $storedCredentialId,
      PublicKeyCredentialDescriptor::CREDENTIAL_TYPE_PUBLIC_KEY,
      json_decode($credentialData['transports'] ?? '[]', TRUE),
      // Assumed attestation type.
      'none',
      new EmptyTrustPath(),
      Uuid::fromString($credentialData['aaguid']),
      $decodedPublicKey,
      $credentialData['user_handle'],
      (int) $credentialData['sign_counter']
    );

    $challengeBinary = Base64UrlSafe::decodeNoPadding($challenge);
    $allowCredentials = [
      PublicKeyCredentialDescriptor::create(
        PublicKeyCredentialDescriptor::CREDENTIAL_TYPE_PUBLIC_KEY,
        $storedCredentialId,
        json_decode($credentialData['transports'] ?? '[]', TRUE)
      ),
    ];

    $userVerification = $userVerificationRequirement === 'required'
      ? PublicKeyCredentialRequestOptions::USER_VERIFICATION_REQUIREMENT_REQUIRED
      : PublicKeyCredentialRequestOptions::USER_VERIFICATION_REQUIREMENT_PREFERRED;

    $timeout = 60000;

    $publicKeyCredentialRequestOptions = PublicKeyCredentialRequestOptions::create(
      $challengeBinary,
      $rpId,
      $allowCredentials,
      $userVerification,
      $timeout
    );

    try {
      $validatedSource = $this->validateAssertion(
        $content,
        $publicKeyCredentialRequestOptions,
        $publicKeyCredentialSource,
        $host,
        $credentialData['user_handle']
      );
    }
    catch (\Throwable $e) {
      // Throw a more descriptive error for logging.
      throw new \Exception('WebAuthn validation failed: ' . $e->getMessage(), 0, $e);
    }

    return TRUE;
  }

}
