<?php

namespace Drupal\wa\Plugin\views\field;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\wa\Service\PasskeyLabeler;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Displays a human-friendly passkey provider label.
 *
 * @ViewsField("wa_passkey_provider")
 */
class PasskeyProvider extends FieldPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Additional fields required from the query.
   *
   * @var string[]
   */
  public $additionalFields = ['aaguid', 'transports'];

  /**
   * Passkey labeler service.
   *
   * @var \Drupal\wa\Service\PasskeyLabeler
   */
  protected $passkeyLabeler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('wa.passkey_labeler')
    );
  }

  /**
   * Constructs a passkey provider field plugin.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, PasskeyLabeler $passkey_labeler) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->passkeyLabeler = $passkey_labeler;
  }

  /**
   * {@inheritdoc}
   */
  public function render($values) {
    $record = (object) [
      'aaguid' => $this->getFieldValueSafe($values, 'aaguid') ?? '',
      'transports' => $this->getFieldValueSafe($values, 'transports'),
    ];

    return [
      '#markup' => $this->passkeyLabeler->getKeyTypeFromRecord($record),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    // Only load the supporting fields we need; no real column for provider.
    $this->addAdditionalFields($this->additionalFields);
  }

  /**
   * Safely fetches an additional field value from the row.
   */
  protected function getFieldValueSafe($values, string $field) {
    $alias = $this->aliases[$field] ?? NULL;
    if ($alias && isset($values->{$alias})) {
      return $values->{$alias};
    }

    $fallback = $this->tableAlias . '_' . $field;
    if (isset($values->{$fallback})) {
      return $values->{$fallback};
    }

    return $values->{$field} ?? NULL;
  }

}
