<?php

namespace Drupal\wa\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Flood\FloodInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\wa\Service\PasskeyLabeler;
use Drupal\wa\Service\WebAuthnService;
use ParagonIE\ConstantTime\Base64UrlSafe;
use Drupal\user\UserInterface;
use Drupal\user\RoleInterface;

/**
 * Controller for Passkey registration, authentication, and management.
 *
 * This controller handles the routes for:
 * - Listing and managing user passkeys.
 * - Generating options for WebAuthn registration and authentication.
 * - Verifying WebAuthn responses for registration and login.
 */
class PasskeyController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The passkey labeler service.
   *
   * @var \Drupal\wa\Service\PasskeyLabeler
   */
  protected $passkeyLabeler;

  /**
   * The WebAuthn service.
   *
   * @var \Drupal\wa\Service\WebAuthnService
   */
  protected $webAuthnService;

  /**
   * The CSRF token generator.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected $csrfToken;

  /**
   * The flood service.
   *
   * @var \Drupal\Core\Flood\FloodInterface
   */
  protected $flood;

  /**
   * Constructs a PasskeyController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\wa\Service\PasskeyLabeler $passkey_labeler
   *   The passkey labeler service.
   * @param \Drupal\wa\Service\WebAuthnService $web_authn_service
   *   The WebAuthn service.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
   *   The CSRF token generator.
   * @param \Drupal\Core\Flood\FloodInterface $flood
   *   The flood service.
   */
  public function __construct(Connection $database, TimeInterface $time, PasskeyLabeler $passkey_labeler, WebAuthnService $web_authn_service, CsrfTokenGenerator $csrf_token, FloodInterface $flood) {
    $this->database = $database;
    $this->time = $time;
    $this->passkeyLabeler = $passkey_labeler;
    $this->webAuthnService = $web_authn_service;
    $this->csrfToken = $csrf_token;
    $this->flood = $flood;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('datetime.time'),
      $container->get('wa.passkey_labeler'),
      $container->get('wa.webauthn'),
      $container->get('csrf_token'),
      $container->get('flood')
    );
  }

  /**
   * Checks if the current user has an allowed role for passkey operations.
   *
   * @return bool
   *   TRUE if the user has an allowed role based on configuration rules, FALSE
   *   otherwise. Selecting the authenticated role allows all authenticated
   *   users; selecting no roles allows nobody.
   */
  protected function currentUserHasAllowedRole(): bool {
    $config = $this->config('wa.settings');
    $allowed_roles = $config->get('allowed_roles') ?? [];
    $allowed_roles = array_filter($allowed_roles);

    if (empty($allowed_roles)) {
      return FALSE;
    }

    // If authenticated role is selected, allow any authenticated user.
    if (in_array(RoleInterface::AUTHENTICATED_ID, $allowed_roles, TRUE)) {
      return $this->currentUser()->isAuthenticated();
    }

    $user_roles = $this->currentUser()->getRoles();
    return !empty(array_intersect($user_roles, $allowed_roles));
  }

  /**
   * Renders the Passkey management page.
   */
  public function managePasskeys(UserInterface $user) {
    $delete_path = ltrim(Url::fromRoute('wa.passkey_delete')->getInternalPath(), '/');
    $delete_token = $this->csrfToken->get($delete_path);
    $register_path = ltrim(Url::fromRoute('wa.register_options')->getInternalPath(), '/');
    $register_token = $this->csrfToken->get($register_path);
    $register_verify_path = ltrim(Url::fromRoute('wa.register_verify')->getInternalPath(), '/');
    $register_verify_token = $this->csrfToken->get($register_verify_path);
    $login_path = ltrim(Url::fromRoute('wa.login_options')->getInternalPath(), '/');
    $login_token = $this->csrfToken->get($login_path);

    $build = [
      '#type' => 'container',
      '#attributes' => ['class' => ['passkey-management']],
      '#attached' => [
        'library' => ['wa/passkey'],
        'drupalSettings' => [
          'wa' => [
            'passkeyDeleteToken' => $delete_token,
            'registerOptionsToken' => $register_token,
            'registerVerifyToken' => $register_verify_token,
            'loginOptionsToken' => $login_token,
            'routes' => [
              'passkeyDelete' => Url::fromRoute('wa.passkey_delete')->toString(),
              'registerOptions' => Url::fromRoute('wa.register_options')->toString(),
              'registerVerify' => Url::fromRoute('wa.register_verify')->toString(),
              'loginOptions' => Url::fromRoute('wa.login_options')->toString(),
              'loginVerify' => Url::fromRoute('wa.login_verify')->toString(),
            ],
          ],
        ],
      ],
    ];

    $build['title'] = [
      '#markup' => '<h2>' . $this->t('Manage Passkeys') . '</h2>',
    ];

    $build['description'] = [
      '#markup' => '<p>' . $this->t('Passkeys allow you to log in securely without a password.') . '</p>',
    ];

    // List existing passkeys.
    $header = [
      'authenticator' => $this->t('Authenticator'),
      'created' => $this->t('Created'),
      'last_used' => $this->t('Last Used'),
      'actions' => $this->t('Actions'),
    ];

    $rows = [];
    $query = $this->database->select('wa', 'c')
      ->fields('c', ['credential_id', 'created', 'last_used', 'aaguid'])
      ->condition('uid', $user->id())
      ->execute();

    foreach ($query as $record) {
      $credentialIdEncoded = base64_encode($record->credential_id);
      $rows[] = [
        $this->passkeyLabeler->getKeyTypeFromRecord($record),
        date('Y-m-d H:i', $record->created),
        date('Y-m-d H:i', $record->last_used),
        [
          'data' => [
            '#type' => 'button',
            '#value' => $this->t('Delete'),
            '#attributes' => [
              'class' => ['button', 'button--danger', 'passkey-delete'],
              'data-credential-id' => $credentialIdEncoded,
            ],
          ],
        ],
      ];
    }

    $build['passkey_management'] = [
      '#type' => 'details',
      '#title' => $this->t('Your Passkeys'),
      '#open' => TRUE,
    ];

    $canRegister = (int) $this->currentUser()->id() === (int) $user->id();

    if (empty($rows)) {
      if ($canRegister) {
        $build['passkey_management']['add_button'] = [
          '#type' => 'button',
          '#value' => $this->t('Add New Passkey'),
          '#attributes' => [
            'id' => 'register-passkey',
            'class' => ['button', 'button--primary'],
          ],
        ];
      }
      $build['passkey_management']['table'] = [
        '#markup' => '<p>' . $this->t('No passkeys registered yet.') . '</p>',
      ];
    }
    else {
      $build['passkey_management']['table'] = [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No passkeys registered yet.'),
      ];
      if ($canRegister) {
        $build['passkey_management']['add_button'] = [
          '#type' => 'button',
          '#value' => $this->t('Add New Passkey'),
          '#attributes' => [
            'id' => 'register-passkey',
            'class' => ['button', 'button--primary'],
          ],
        ];
      }
    }

    $build['passkey_management']['status'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'passkey-status'],
    ];

    return $build;
  }

  /**
   * Provides registration options for passkey creation.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The registration options response.
   */
  public function registerOptions(Request $request) {
    $token = $request->headers->get('X-CSRF-Token');
    $path = ltrim($request->getPathInfo(), '/');

    $config = $this->config('wa.settings');
    if ($config->get('enable_passkey_login') === FALSE) {
      return new JsonResponse(['status' => 'error', 'message' => 'Passkey login is disabled.'], 403);
    }

    $valid1 = $this->csrfToken->validate($token, $path);
    $valid2 = $this->csrfToken->validate($token, '/' . $path);

    if (!$valid1 && !$valid2) {
      $this->getLogger('wa')->error('CSRF Fail: Token=@token, Path=@path, Valid1=@v1, Valid2=@v2', [
        '@token' => $token ?? '',
        '@path' => $path ?? '',
        '@v1' => $valid1 ? 'YES' : 'NO',
        '@v2' => $valid2 ? 'YES' : 'NO',
      ]);
      return new JsonResponse(['status' => 'error', 'message' => 'Invalid CSRF token.'], 403);
    }

    if (!$this->currentUserHasAllowedRole()) {
      return new JsonResponse(['status' => 'error', 'message' => 'Passkey registration not permitted for your role.'], 403);
    }

    $user = $this->entityTypeManager()->getStorage('user')->load($this->currentUser()->id());
    if (!$user instanceof UserInterface) {
      return new JsonResponse(['status' => 'error', 'message' => 'Unable to process request.'], 400);
    }
    $rpId = $request->getHost();

    $options = $this->webAuthnService->getRegistrationOptions($user, $rpId);

    return new JsonResponse($options);
  }

  /**
   * Persists a newly registered credential.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request containing the credential response.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The verification result response.
   */
  public function registerVerify(Request $request) {
    $floodName = 'wa_register';
    // 1 hour
    $window = 3600;
    $maxAttempts = 5;
    $ipIdentifier = 'ip:' . ($request->getClientIp() ?? 'unknown');
    $userIdentifier = 'user:' . $this->currentUser()->id();

    if (!$this->flood->isAllowed($floodName, $maxAttempts, $window, $ipIdentifier)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Too many registration attempts. Please try again later.',
      ], 429);
    }
    if (!$this->flood->isAllowed($floodName, $maxAttempts, $window, $userIdentifier)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Too many registration attempts. Please try again later.',
      ], 429);
    }

    try {
      $token = $request->headers->get('X-CSRF-Token');
      $path = ltrim($request->getPathInfo(), '/');

      $valid1 = $this->csrfToken->validate($token, $path);
      $valid2 = $this->csrfToken->validate($token, '/' . $path);

      if (!$valid1 && !$valid2) {
        $this->getLogger('wa')->error('CSRF Fail on verify: Token=@token, Path=@path, Valid1=@v1, Valid2=@v2', [
          '@token' => $token ?? '',
          '@path' => $path ?? '',
          '@v1' => $valid1 ? 'YES' : 'NO',
          '@v2' => $valid2 ? 'YES' : 'NO',
        ]);
        return new JsonResponse(['status' => 'error', 'message' => 'Invalid CSRF token.'], 403);
      }

      $config = $this->config('wa.settings');
      if ($config->get('enable_passkey_login') === FALSE) {
        return new JsonResponse(['status' => 'error', 'message' => 'Passkey login is disabled.'], 403);
      }

      if (!$this->currentUserHasAllowedRole()) {
        return new JsonResponse([
          'status' => 'error',
          'message' => 'Passkey registration not permitted for your role.',
        ], 403);
      }

      $content = json_decode($request->getContent(), TRUE);
      $user = $this->entityTypeManager()->getStorage('user')->load($this->currentUser()->id());
      if (!$user instanceof UserInterface) {
        throw new \Exception('User not found.');
      }
      $rpId = $request->getHost();
      $challenge = $request->getSession()->get('wa_challenge');

      if (empty($challenge)) {
        throw new \Exception('No challenge found in session.');
      }

      // Reconstruct creation options for validation.
      $publicKeyCredentialSource = $this->webAuthnService->validateRegistrationRequest(
        $content,
        $rpId,
        $challenge,
        $request->getHost(),
        $user->getAccountName(),
        (string) $user->id(),
        $user->getDisplayName()
      );
      if (!$publicKeyCredentialSource) {
        throw new \Exception('Passkey registration validation failed.');
      }

      $aaguid = strtolower($publicKeyCredentialSource->aaguid->toRfc4122());
      if (!$this->passkeyLabeler->isAaguidAllowed($aaguid)) {
        $this->getLogger('wa')->warning('Passkey registration blocked for disallowed AAGUID: @aaguid', ['@aaguid' => $aaguid]);
        return new JsonResponse([
          'status' => 'error',
          'message' => 'This authenticator is not allowed for passkey login.',
        ], 403);
      }

      $credentialHash = hash('sha256', $publicKeyCredentialSource->publicKeyCredentialId);

      // Check for duplicate credential ID.
      $exists = $this->database->select('wa', 'c')
        ->fields('c', ['id'])
        ->condition('credential_hash', $credentialHash)
        ->range(0, 1)
        ->execute()
        ->fetchField();

      if ($exists) {
        throw new \Exception('This passkey is already registered.');
      }

      // Store in DB.
      $timestamp = $this->time->getRequestTime();
      $this->database->insert('wa')
        ->fields([
          'uid' => $this->currentUser()->id(),
          'credential_id' => $publicKeyCredentialSource->publicKeyCredentialId,
      // Store as base64 for safe storage.
          'public_key' => base64_encode($publicKeyCredentialSource->credentialPublicKey),
          'credential_hash' => $credentialHash,
          'user_handle' => $publicKeyCredentialSource->userHandle,
          'sign_counter' => $publicKeyCredentialSource->counter,
          'transports' => json_encode($publicKeyCredentialSource->transports),
          'aaguid' => $publicKeyCredentialSource->aaguid->toRfc4122(),
          'created' => $timestamp,
          'last_used' => $timestamp,
        ])
        ->execute();

      $encodedCredentialId = base64_encode($publicKeyCredentialSource->publicKeyCredentialId);
      $maskedCredentialId = '****' . substr($encodedCredentialId, -6);
      $this->getLogger('wa')->info('Passkey added for user @uid. Credential ID: @credential_id, AAGUID: @aaguid.', [
        '@uid' => $this->currentUser()->id(),
        '@credential_id' => $maskedCredentialId,
        '@aaguid' => $publicKeyCredentialSource->aaguid->toRfc4122(),
      ]);

      return new JsonResponse(['status' => 'ok']);
    }
    catch (\Throwable $e) {
      $this->flood->register($floodName, $window, $ipIdentifier);
      $this->flood->register($floodName, $window, $userIdentifier);

      $this->getLogger('wa')->error('Passkey registration failed: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse(['status' => 'error', 'message' => 'Passkey registration failed.'], 400);
    }
    finally {
      $request->getSession()->remove('wa_challenge');
    }
  }

  /**
   * Deletes a stored passkey for the current user (or admin).
   */
  public function deletePasskey(Request $request) {
    $token = $request->headers->get('X-CSRF-Token');
    if (!$this->csrfToken->validate($token, 'wa/passkey/delete')) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Invalid CSRF token.',
      ], 403);
    }

    $data = json_decode($request->getContent(), TRUE) ?? [];
    $credentialId = $data['id'] ?? $request->request->get('id');

    if (empty($credentialId)) {
      return new JsonResponse(['status' => 'error', 'message' => 'Missing credential id'], 400);
    }

    $canAdminister = $this->currentUser()->hasPermission('administer all user passkey');
    if (!$canAdminister) {
      // Check if user has allowed role for API endpoint consistency.
      if (!$this->currentUserHasAllowedRole()) {
        return new JsonResponse(['status' => 'error', 'message' => 'Deletion not permitted.'], 403);
      }
    }

    $current_uid = (int) $this->currentUser()->id();

    // Accept base64-encoded IDs from the UI and raw IDs from other callers.
    $candidateIds = [$credentialId];
    $decodedBase64 = base64_decode($credentialId, TRUE);
    if ($decodedBase64 !== FALSE && $decodedBase64 !== '') {
      $candidateIds[] = $decodedBase64;
    }
    try {
      $decodedUrl = Base64UrlSafe::decodeNoPadding($credentialId);
      if ($decodedUrl !== '') {
        $candidateIds[] = $decodedUrl;
      }
    }
    catch (\Throwable $e) {
      // Ignore decode failures and continue with other candidates.
    }
    $candidateIds = array_values(array_unique(array_filter($candidateIds, static function ($value) {
      return $value !== '';
    })));

    $database = $this->database;

    // Fetch record details before deletion so we can log the owner correctly.
    $query = $database->select('wa', 'c')
      ->fields('c', ['aaguid', 'uid'])
      ->condition('credential_id', $candidateIds, 'IN');
    $result = $query->execute()->fetchAssoc();
    $target_uid = $result ? (int) $result['uid'] : NULL;
    $aaguid = $result ? $result['aaguid'] : 'Unknown';

    // Delete for owner.
    $deleted = $this->deleteByCredentialIds($candidateIds, $current_uid);

    // Allow administrators with permission to remove any credential.
    if (!$deleted && $this->currentUser()->hasPermission('administer all user passkey')) {
      $deleted = $this->deleteByCredentialIds($candidateIds);
    }

    if ($deleted) {
      $this->getLogger('wa')->info('Passkey deleted for user @uid. Credential ID: @credential_id, AAGUID: @aaguid.', [
        '@credential_id' => base64_encode($credentialId),
        '@uid' => $target_uid ?? $current_uid,
        '@aaguid' => $aaguid,
      ]);
      return new JsonResponse(['status' => 'ok']);
    }

    return new JsonResponse(['status' => 'error', 'message' => 'Deletion failed or unauthorized'], 403);
  }

  /**
   * Deletes credentials matching any of the provided IDs.
   */
  protected function deleteByCredentialIds(array $candidateIds, ?int $uid = NULL): int {
    if (empty($candidateIds)) {
      return 0;
    }

    $query = $this->database->delete('wa')
      ->condition('credential_id', $candidateIds, 'IN');

    if ($uid !== NULL) {
      $query->condition('uid', $uid);
    }

    return (int) $query->execute();
  }

  /**
   * Provides login options for passkey authentication.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The login options response.
   */
  public function loginOptions(Request $request) {
    $floodName = 'wa_login_options';
    $window = 300;
    $maxAttempts = 10;
    $ipIdentifier = 'ip:' . ($request->getClientIp() ?? 'unknown');

    // Check the Origin/Referer header.
    $config = $this->config('wa.settings');
    if ($config->get('enable_passkey_login') === FALSE) {
      return new JsonResponse(['status' => 'error', 'message' => 'Passkey login is disabled.'], 403);
    }

    if (!$this->flood->isAllowed($floodName, $maxAttempts, $window, $ipIdentifier)) {
      return new JsonResponse(['status' => 'error', 'message' => 'Too many login attempts. Please try again later.'], 429);
    }

    $origin = $request->headers->get('Origin');
    $referer = $request->headers->get('Referer');
    $host = $request->getHost();

    // Allow if Origin matches host (standard for CORS/fetch).
    // If Origin is missing, check Referer.
    $valid_origin = FALSE;
    if ($origin) {
      $parsed_origin = parse_url($origin);
      if (isset($parsed_origin['host']) && $parsed_origin['host'] === $host) {
        $valid_origin = TRUE;
      }
    }
    elseif ($referer) {
      $parsed_referer = parse_url($referer);
      if (isset($parsed_referer['host']) && $parsed_referer['host'] === $host) {
        $valid_origin = TRUE;
      }
    }

    if (!$valid_origin) {
      return new JsonResponse(['status' => 'error', 'message' => 'Invalid Origin or Referer.'], 403);
    }

    $rpId = $request->getHost();
    $challenge = random_bytes(32);
    $challengeEncoded = Base64UrlSafe::encodeUnpadded($challenge);
    $request->getSession()->set('wa_challenge', $challengeEncoded);

    $userVerification = $config->get('user_verification') ?? 'preferred';

    $options = [
      'challenge' => $challengeEncoded,
      'timeout' => 60000,
      'rpId' => $rpId,
      'userVerification' => $userVerification,
    ];

    return new JsonResponse($options);
  }

  /**
   * Validates a passkey assertion and logs the user in.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request containing the assertion response.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The verification result response.
   */
  public function loginVerify(Request $request) {
    $floodName = 'wa_login';
    $window = 900;
    $maxAttempts = 5;
    $ipIdentifier = 'ip:' . ($request->getClientIp() ?? 'unknown');
    $credentialIdentifier = NULL;
    $userIdentifier = NULL;
    $registerFloodOnFailure = TRUE;

    if (!$this->flood->isAllowed($floodName, $maxAttempts, $window, $ipIdentifier)) {
      return new JsonResponse(['status' => 'error', 'message' => 'Too many login attempts. Please try again later.'], 429);
    }

    try {
      $config = $this->config('wa.settings');
      if ($config->get('enable_passkey_login') === FALSE) {
        return new JsonResponse(['status' => 'error', 'message' => 'Passkey login is disabled.'], 403);
      }

      $content = json_decode($request->getContent(), TRUE);
      $credentialId = $content['id'] ?? '';
      if ($credentialId === '') {
        throw new \Exception('Missing credential id.');
      }

      $credentialIdentifier = 'cred:' . hash('sha256', $credentialId);
      if (!$this->flood->isAllowed($floodName, $maxAttempts, $window, $credentialIdentifier)) {
        return new JsonResponse(['status' => 'error', 'message' => 'Too many login attempts. Please try again later.'], 429);
      }

      $candidateIds = [$credentialId];
      $decodedBase64 = base64_decode($credentialId, TRUE);
      if ($decodedBase64 !== FALSE && $decodedBase64 !== '') {
        $candidateIds[] = $decodedBase64;
      }
      try {
        $decodedUrl = Base64UrlSafe::decodeNoPadding($credentialId);
        if ($decodedUrl !== '') {
          $candidateIds[] = $decodedUrl;
        }
      }
      catch (\Throwable $e) {
        // Ignore decode failures; we'll fall back to the raw value.
      }
      $candidateIds = array_values(array_unique(array_filter($candidateIds, static function ($value) {
        return $value !== '';
      })));

      $challenge = $request->getSession()->get('wa_challenge');

      if (empty($challenge)) {
        throw new \Exception('No challenge found in session.');
      }

      // Find user by credential ID.
      $database = $this->database;
      $query = $database->select('wa', 'c')
        ->fields('c', ['uid', 'credential_id', 'public_key', 'user_handle', 'sign_counter', 'transports', 'aaguid'])
        ->condition('credential_id', $candidateIds, 'IN');

      // If the client provided the user handle (user ID), use it
      // to narrow down the search.
      $userHandleEncoded = $content['response']['userHandle'] ?? '';
      $enforce_user_handle = $config->get('enforce_user_handle');

      // Check if we must enforce user handle presence.
      if ($enforce_user_handle && $userHandleEncoded === '') {
        throw new \Exception('User handle is required by policy.');
      }

      if ($userHandleEncoded !== '') {
        $userHandle = '';
        try {
          $userHandle = Base64UrlSafe::decodeNoPadding($userHandleEncoded);
        }
        catch (\Throwable $e) {
          if ($enforce_user_handle) {
            // When enforcement is enabled, reject malformed user handles.
            throw new \Exception('User handle decode failed.');
          }
          // When enforcement is disabled, ignore invalid user handle encoding.
          // This is just an optimization hint, so we fall back to the standard
          // lookup by credential_id.
        }

        if ($userHandle !== '') {
          // Add uid condition with AND logic.
          $query->condition('uid', $userHandle);
        }
        elseif ($enforce_user_handle) {
          // If we must enforce user handle presence,
          // The user ID must be matched.
          throw new \Exception('User handle is required by policy.');
        }
      }

      $statement = $query->execute();

      $result = $statement->fetchAssoc();

      if (!$result) {
        throw new \Exception('Credential not found.');
      }

      $uid = (int) $result['uid'];
      $user = $this->entityTypeManager()->getStorage('user')->load($uid);
      $userIdentifier = 'user:' . $uid;

      if (!$user instanceof UserInterface) {
        throw new \Exception('User not found or invalid.');
      }

      if (!$user->isActive()) {
        throw new \Exception('User account is blocked.');
      }

      // Check if the user is allowed to login with passkey.
      $allowed_roles = $config->get('allowed_roles') ?? [];
      $allowed_roles = array_filter($allowed_roles);
      if (empty($allowed_roles)) {
        // If no roles are configured, no one is allowed to use passkeys.
        throw new \Exception('Passkey login not permitted. No roles are configured.');
      }

      $user_roles = $user->getRoles();
      if (empty(array_intersect($user_roles, $allowed_roles))) {
        throw new \Exception('Passkey login not permitted for this account role.');
      }

      if (!$this->passkeyLabeler->isAaguidAllowed(strtolower((string) $result['aaguid']))) {
        throw new \Exception('This authenticator is not allowed for passkey login.');
      }

      // Validate assertion via WebAuthn service.
      $storedCredentialId = $result['credential_id'];
      $storedPublicKey = $result['public_key'] ?? '';

      $publicKeyCredentialSource = NULL;
      $validated = $this->webAuthnService->validateLoginRequest(
        $content,
        [
          'credential_id' => $storedCredentialId,
          'transports' => $result['transports'] ?? '[]',
          'aaguid' => $result['aaguid'],
          'public_key' => $storedPublicKey,
          'user_handle' => $result['user_handle'],
          'sign_counter' => $result['sign_counter'],
        ],
        $challenge,
        $request->getHost(),
        $request->getHost(),
        $config->get('user_verification') ?? 'preferred',
        $publicKeyCredentialSource
      );
      if ($validated !== TRUE || !$publicKeyCredentialSource) {
        throw new \Exception('Passkey validation failed.');
      }

      // Update counter and last used.
      $database->update('wa')
        ->fields([
          'last_used' => $this->time->getRequestTime(),
          'sign_counter' => $publicKeyCredentialSource->counter,
        ])
        ->condition('credential_id', $storedCredentialId)
        ->execute();

      user_login_finalize($user);
      $maskedCredentialId = '****' . substr($credentialId, -6);
      $this->getLogger('wa')->info('User logged in with passkey. User: @uid, Credential ID: @credential_id.', [
        '@uid' => $uid,
        '@credential_id' => $maskedCredentialId,
      ]);

      $registerFloodOnFailure = FALSE;
      return new JsonResponse(['status' => 'ok', 'uid' => $uid]);
    }
    catch (\Throwable $e) {
      if ($registerFloodOnFailure) {
        $this->flood->register($floodName, $window, $ipIdentifier);
        if ($credentialIdentifier) {
          $this->flood->register($floodName, $window, $credentialIdentifier);
        }
        if ($userIdentifier) {
          $this->flood->register($floodName, $window, $userIdentifier);
        }
      }

      $this->getLogger('wa')->error('Passkey login failed: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse(['status' => 'error', 'message' => 'Passkey login failed.'], 400);
    }
    finally {
      // Clear the challenge to enforce one-time use.
      $request->getSession()->remove('wa_challenge');
    }
  }

}
