<?php

namespace Drupal\wa\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\user\Entity\User;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Checks access for passkey management.
 */
class PasskeyManageAccessCheck implements AccessInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new PasskeyManageAccessCheck object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public function access(RouteMatchInterface $route_match, AccountInterface $account): AccessResult {
    $user_param = $route_match->getParameter('user');

    if ($user_param instanceof User) {
      $target_uid = (int) $user_param->id();
    }
    elseif (is_numeric($user_param)) {
      $target_uid = (int) $user_param;
    }
    else {
      return AccessResult::forbidden()->cachePerUser();
    }

    // Global switch.
    $config = $this->configFactory->get('wa.settings');
    if ($config->get('enable_passkey_login') === FALSE) {
      return AccessResult::forbidden()->addCacheableDependency($config);
    }

    // Anonymous users cannot manage passkeys.
    if (!$account->isAuthenticated()) {
      return AccessResult::forbidden()->cachePerUser();
    }

    $canAdminister = $account->hasPermission('administer all user passkey');

    // Owners can manage their own passkeys.
    if ((int) $account->id() === $target_uid) {
      // Check allowed roles.
      $allowed_roles = $config->get('allowed_roles') ?? [];
      // Filter out empty values just in case.
      $allowed_roles = array_filter($allowed_roles);

      if (!empty($allowed_roles)) {
        $user_roles = $account->getRoles();
        if (empty(array_intersect($user_roles, $allowed_roles))) {
          return AccessResult::forbidden()->addCacheableDependency($config)->cachePerUser();
        }
      }

      return AccessResult::allowed()->addCacheableDependency($config)->cachePerUser();
    }

    // Administrators with the custom permission can manage all passkeys.
    if ($canAdminister) {
      return AccessResult::allowed()->cachePerPermissions();
    }

    return AccessResult::forbidden()->cachePerUser()->cachePerPermissions();
  }

}
