<?php

namespace Drupal\Tests\watchdog_mailer\Functional;

/**
 * This class provides methods specifically for testing something.
 *
 * @group watchdog_mailer
 */
class WatchdogMailerSettingsTest extends WatchdogMailerTestBase {

  /**
   * Tests initial values.
   */
  public function testInitialValues() {
    $session = $this->assertSession();
    // Go to the settings page and check if the default settings are applied.
    $this->drupalGet('/admin/config/development/watchdog_mailer');
    $session->checkboxChecked('enabled');
    $session->fieldValueEquals('recipients_default', '');
    $session->fieldValueEquals('mail_subject', "[Watchdog Mailer ([watchdog_mailer:channel])] \"[watchdog_mailer:message]\" at [site:name] ([site:url])");
    $session->fieldValueEquals('mail_body', "[site:name] ([site:url]) reported the following:\n\n[watchdog_mailer:message]\n[watchdog_mailer:php_backtrace]\n\n---\n\nSite: [site:url]\nLocation: [watchdog_mailer:request_uri]\nChannel: [watchdog_mailer:channel]\nBy: [watchdog_mailer:user:display-name] (UID: [watchdog_mailer:user], IP: [watchdog_mailer:user_ip])\nAt: [watchdog_mailer:timestamp:medium]\nReferer: [watchdog_mailer:referer]\n\n---\nThis message was generated by Watchdog Mailer\n[watchdog_mailer:watchdog_mailer_url]");
    $session->fieldValueEquals('mail_limit', 0);
    $session->fieldValueEquals('mail_limit_time_frame', 3600);
    $session->fieldValueEquals('limit_mail_subject', '[Watchdog Mailer] Mail limit reached at [site:name] ([site:url])');
    $session->fieldValueEquals('limit_mail_body', "[site:name] ([site:url]) reported the following:\n\nWatchdog Mailer has reached its mail limit for [site:name].\nPlease check your error log for further information.");
    $session->checkboxChecked('notification_objects[0][enabled]');
    $session->fieldValueEquals('notification_objects[0][channels]', '');
    $session->checkboxNotChecked('notification_objects[0][channels_negate]');
    $session->checkboxNotChecked('notification_objects[0][severities][rfc0]');
    $session->checkboxNotChecked('notification_objects[0][severities][rfc1]');
    $session->checkboxNotChecked('notification_objects[0][severities][rfc2]');
    $session->checkboxNotChecked('notification_objects[0][severities][rfc3]');
    $session->checkboxNotChecked('notification_objects[0][severities][rfc4]');
    $session->checkboxNotChecked('notification_objects[0][severities][rfc5]');
    $session->checkboxNotChecked('notification_objects[0][severities][rfc6]');
    $session->checkboxNotChecked('notification_objects[0][severities][rfc7]');
    $session->fieldValueEquals('notification_objects[0][recipients]', '');
  }

  /**
   * Tests the basic functionality of the settings page.
   */
  public function testBasicSettingsChange() {
    $session = $this->assertSession();
    // Go to the settings page and change all the settings.
    $this->drupalGet('/admin/config/development/watchdog_mailer');
    $edit = [
      'enabled' => FALSE,
      'recipients_default' => 'foo@bar.baz,x@y.z',
      'mail_subject' => 'test_mail_subject',
      'mail_body' => 'test_mail_body',
      'mail_limit' => 456,
      'mail_limit_time_frame' => 789,
      'limit_mail_subject' => 'test_limit_mail_subject',
      'limit_mail_body' => 'test_limit_mail_body',
      'notification_objects[0][enabled]' => FALSE,
      'notification_objects[0][channels]' => "test_channelA\ntest_channelB",
      'notification_objects[0][channels_negate]' => TRUE,
      'notification_objects[0][severities][rfc0]' => TRUE,
      'notification_objects[0][severities][rfc1]' => TRUE,
      'notification_objects[0][severities][rfc2]' => TRUE,
      'notification_objects[0][severities][rfc3]' => TRUE,
      'notification_objects[0][severities][rfc4]' => TRUE,
      'notification_objects[0][severities][rfc5]' => TRUE,
      'notification_objects[0][severities][rfc6]' => TRUE,
      'notification_objects[0][severities][rfc7]' => TRUE,
      'notification_objects[0][recipients]' => 'baz@foo.bar, a@b.c',
    ];
    $this->submitForm($edit, 'Save configuration');
    // Check saved settings.
    $session->pageTextContains('The configuration options have been saved.');
    $session->checkboxNotChecked('enabled');
    $session->fieldValueEquals('recipients_default', 'foo@bar.baz,x@y.z');
    $session->fieldValueEquals('mail_subject', 'test_mail_subject');
    $session->fieldValueEquals('mail_body', 'test_mail_body');
    $session->fieldValueEquals('mail_limit', 456);
    $session->fieldValueEquals('mail_limit_time_frame', 789);
    $session->fieldValueEquals('limit_mail_subject', 'test_limit_mail_subject');
    $session->fieldValueEquals('limit_mail_body', 'test_limit_mail_body');
    $session->checkboxNotChecked('notification_objects[0][enabled]');
    $session->fieldValueEquals('notification_objects[0][channels]', "test_channelA\ntest_channelB");
    $session->checkboxChecked('notification_objects[0][channels_negate]');
    $session->checkboxChecked('notification_objects[0][severities][rfc0]');
    $session->checkboxChecked('notification_objects[0][severities][rfc1]');
    $session->checkboxChecked('notification_objects[0][severities][rfc2]');
    $session->checkboxChecked('notification_objects[0][severities][rfc3]');
    $session->checkboxChecked('notification_objects[0][severities][rfc4]');
    $session->checkboxChecked('notification_objects[0][severities][rfc5]');
    $session->checkboxChecked('notification_objects[0][severities][rfc6]');
    $session->checkboxChecked('notification_objects[0][severities][rfc7]');
    $session->fieldValueEquals('notification_objects[0][recipients]', 'baz@foo.bar,a@b.c');
    // Check configuration.
    $this->rebuildContainer();
    $watchdogMailerSettings = $this->config('watchdog_mailer.settings');
    $this->assertEquals($watchdogMailerSettings->get('enabled'), FALSE);
    $this->assertEquals($watchdogMailerSettings->get('recipients_default'), ['foo@bar.baz', 'x@y.z']);
    $this->assertEquals($watchdogMailerSettings->get('mail_subject'), 'test_mail_subject');
    $this->assertEquals($watchdogMailerSettings->get('mail_body'), 'test_mail_body');
    $this->assertEquals($watchdogMailerSettings->get('mail_limit'), 456);
    $this->assertEquals($watchdogMailerSettings->get('mail_limit_time_frame'), 789);
    $this->assertEquals($watchdogMailerSettings->get('limit_mail_subject'), 'test_limit_mail_subject');
    $this->assertEquals($watchdogMailerSettings->get('limit_mail_body'), 'test_limit_mail_body');
    $this->assertEquals($watchdogMailerSettings->get('notification_objects.0.enabled'), FALSE);
    $this->assertEquals($watchdogMailerSettings->get('notification_objects.0.channels'), [
      'test_channelA', 'test_channelB',
    ]);
    $this->assertEquals($watchdogMailerSettings->get('notification_objects.0.channels_negate'), TRUE);
    $this->assertEquals($watchdogMailerSettings->get('notification_objects.0.severities'), [
      0 => 0,
      1 => 1,
      2 => 2,
      3 => 3,
      4 => 4,
      5 => 5,
      6 => 6,
      7 => 7,
    ]);
    $this->assertEquals($watchdogMailerSettings->get('notification_objects.0.recipients'), ['baz@foo.bar', 'a@b.c']);
  }

  /**
   * Tests adding and removing notification objects via the form.
   */
  public function testNotificationObjectsAddRemove() {
    $session = $this->assertSession();
    // Set a value in object #0 and add a new one.
    $this->drupalGet('/admin/config/development/watchdog_mailer');
    $edit = [
      'enabled' => TRUE,
      'notification_objects[0][channels]' => 'notification_object_a',
    ];
    $this->submitForm($edit, 'Add Notification Type');
    $this->rebuildContainer();
    // Check that the old value is still there and modify values in #0 and #1.
    $session->fieldValueEquals('notification_objects[0][channels]', 'notification_object_a');
    $session->fieldExists('notification_objects[1][channels]');
    $edit = [
      'notification_objects[0][channels]' => 'notification_object_a_changed',
      'notification_objects[1][channels]' => 'notification_object_b',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->rebuildContainer();
    // Check that all changes are applied correctly.
    $session->fieldValueEquals('notification_objects[0][channels]', 'notification_object_a_changed');
    $session->fieldValueEquals('notification_objects[1][channels]', 'notification_object_b');
    // Remove object #0 and see that #1 has taken its place now.
    $edit = [
      'notification_objects[0][remove]' => 'TRUE',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->rebuildContainer();
    $session->fieldValueEquals('notification_objects[0][channels]', 'notification_object_b');
    $session->fieldNotExists('notification_objects[1][channels]');
  }

  /**
   * Tests settings form error functionality.
   */
  public function testSettingsFormErrors() {
    $this->drupalGet('/admin/config/development/watchdog_mailer');
    // Test the recipients_default field.
    // Test single invalid email-address.
    $edit = [
      'enabled' => TRUE,
      'recipients_default' => 'invalid_email_address',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->statusMessageContains('Invalid email address(es) given.', 'error');
    // Test multiple invalid email-address.
    $edit = [
      'enabled' => TRUE,
      'recipients_default' => 'example@example.com, foobar.baz_invalid',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->statusMessageContains('Invalid email address(es) given.', 'error');
    // Test invalid comma separation.
    $edit = [
      'enabled' => TRUE,
      'recipients_default' => 'example@example.com, foobar@baz.bar,',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->statusMessageContains('Invalid email address(es) given.', 'error');
    // Do the same for the recipients field of the default notification object.
    // Test single invalid email-address.
    $edit = [
      'enabled' => TRUE,
      'notification_objects[0][recipients]' => 'invalid_email_address',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->statusMessageContains('Invalid email address(es) given.', 'error');
    // Test multiple invalid email-address.
    $edit = [
      'enabled' => TRUE,
      'notification_objects[0][recipients]' => 'example@example.com, foobar.baz_invalid',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->statusMessageContains('Invalid email address(es) given.', 'error');
    // Test invalid comma separation.
    $edit = [
      'enabled' => TRUE,
      'notification_objects[0][recipients]' => 'example@example.com, foobar@baz.bar,',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->statusMessageContains('Invalid email address(es) given.', 'error');
    // Create a second notification object and test the same.
    $this->drupalGet('/admin/config/development/watchdog_mailer');
    $this->submitForm([], 'Add Notification Type');
    $this->rebuildContainer();
    // Test single invalid email-address.
    $edit = [
      'enabled' => TRUE,
      'notification_objects[1][recipients]' => 'invalid_email_address',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->statusMessageContains('Invalid email address(es) given.', 'error');
    // Test multiple invalid email-address.
    $edit = [
      'enabled' => TRUE,
      'notification_objects[1][recipients]' => 'example@example.com, foobar.baz_invalid',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->statusMessageContains('Invalid email address(es) given.', 'error');
    // Test invalid comma separation.
    $edit = [
      'enabled' => TRUE,
      'notification_objects[1][recipients]' => 'example@example.com, foobar@baz.bar,',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->statusMessageContains('Invalid email address(es) given.', 'error');
  }

}
