<?php

/**
 * @file
 * Unsurprisingly contains Drupal\wayback\Plugin\Filter\wayback_filter.
 */
namespace Drupal\wayback_filter\Plugin\Filter;

use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * @Filter(
 *   id = "wayback_filter",
 *   title = @Translation("Wayback Filter"),
 *   description = @Translation("Add Wayback links to existing links in older nodes."),
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_MARKUP_LANGUAGE,
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_TRANSFORM_IRREVERSIBLE,
 * )
 */
class wayback_filter extends FilterBase {

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode) {
    $config = \Drupal::config('wayback_filter.settings');
    // v($text, 0 , 'body text');
    $result = new FilterProcessResult($text); // If we have to return something unaltered
    $arg = explode('/', \Drupal\Core\Url::fromRoute('<current>')->getInternalPath()); // Give me back my old functions

    // If not showing a node in full, simply return
    if (empty($arg[1]) || !empty($arg[2]) && ($arg[1] != 'node' && !is_numeric($arg[1]))) {
      return $result;
    }

    // Get the age of the node
    $created = \Drupal::database()->select('node_field_data', 'n')->fields('n', ['created]'])->condition('nid', $arg[1])->execute()->fetchAssoc();

    // When to employ Wayback links?
    // Always
    $waybacklink_start_date = 2524608061; // Sat Jan 01 2050 00:01:01
    if ($config->get('waybacklink_start') > 0) {
      // Unless someone has requested that the node be a number of years old
      $waybacklink_start_date = strtotime('-' . $config->get('waybacklink_start') . ' years');
    }

    // Don't bother with new nodes
    if (empty($created['created']) || $created['created'] > $waybacklink_start_date) {
      // v('too new');
      return $result;
    }

    // Find <a codes
    $regexp = "<a\s[^>]*href=(\"??)([^\" >]*?)\\1[^>]*>(.*)<\/a>";
    preg_match_all("/$regexp/siU", $text, $links); // stuff all the links in the text into $links array
    $done = array();
    // v($links,0,'links in the body text');

    // Go through each
    foreach ($links[0] as $orig_code) {
      // Find hrefs
      preg_match('/href="([^"]*)"/i', $orig_code, $href);

      if (substr($href[1], 0, 22) == 'https://web.archive.org') {
        // Do not bother waybacking Wayback links
        continue;
      }

      if ($href && !isset($done[$href[1]])) {
        // Work on http and https refs only
        if (substr($href[1], 0, 7) == 'http://' || substr($href[1], 0, 8) == 'https://') {
          // Create wayback link
          $time = date('YmdHis', $created['created']); // $node->created
          $wayback_link = 'https://web.archive.org/web/' . $time . '/' . $href[1];
          $wayback = $orig_code . ' <a title="' . $config->get('waybacklink_title') .'" class="wayback-link" href="' . $wayback_link . '"><span class = "waybacklink_icon">' . $config->get('waybacklink_icon') . '</span></a>';
          // Replace with added Wayback-link
          $text = str_replace($orig_code, $wayback, $text);
        }
        // Avoid repeated replaces
        $done[$href[1]] = TRUE;
      }
    }
      $result = new FilterProcessResult($text);
      return $result;
  }

}

/*


Please, I don't want this module to kill my site
It will not kill your site as long as your database tables are utf8mb4 encoded. 99,5% of all Drupal 8, 9 and 10 installation run on utf8mb4 databases. So you should have no problems.

How do I install it?
As any other module.

How do I set it up well?
1. Go to the config page (/admin/config/content/wayback_filter) and decide when the links should appear and how they should look. Remember to save.
2. Go to the config for a text format (/admin/config/content/formats), enable Wayback filter and place it as one of the last filters in the <em>Filter processing order</em>. Remember to save.
3. If you're very keen to see it in action, you can set the "Place Wayback links on nodes older than..." to 0 years. Then all links will have trailing Wayback Links.
4. Adding Wayback links to link fields is experimental. Read about it later in this file.

How do I test it?
1. Go to a very old node (with some links in the body) employing the text format you've just set up. It'll show Wayback Links trailing the existing links.
2. Rejoice or
2. Flush your caches and rejoice (with Drupal 8+ you can never flush your caches too often).

What if the Wayback links do not show?
1. The node you're looking at is maybe not that old? Edit it to find out. If you wish, you can temporarily change the date of the node to 01.01.2000.
2. The node might not be using the text format you just spent some time setting up so well with Wayback links. Choose another node or make a new node with an old date from scratch.

How do I make the trailing Wayback links look even more sexy?
Add <em>.wayback-link a {font-size: 4rem;}</em> somewhere in your stylesheet. Flush some caches. Using the Wayback filter settings form and css, you should be able to make it look awesome.

The great <a  href = "https://www.drupal.org/project/extlink">External Links module</a> adds an icon to my Wayback links. I don't want that!
Add <em>.wayback-link svg {display: none;}</em> to your style sheet. Another (and most probably inferior) way is to go to /admin/config/user-interface/extlink click <em>Pattern Matching</em> and, under <em>Exclude links matching the pattern</em>, write <em>web\.archive\.org</em>.

Can I add Wayback Links to my link fields?
If you're able to write a correct machine name on the settings page, you might have some luck.

*/
