<?php

namespace Drupal\webdam_sns\EventSubscriber;

use Drupal\amazon_sns\Event\SnsEvents;
use Drupal\amazon_sns\Event\SnsMessageEvent;
use Drupal\webdam\WebdamConnector;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\webdam\WebdamService;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribes to Amazon SNS notifications to update WebDAM metadata.
 */
class NotificationSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * The WebDAM service.
   *
   * @var \Drupal\webdam\WebdamService
   */
  protected $webdamService;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * NotifcationSubscriber constructor.
   *
   * @param \Drupal\webdam\WebdamService $webdam_service
   *   The WebDAM connector.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   */
  public function __construct(WebdamService $webdam_service, EntityTypeManagerInterface $entity_type_manager, LoggerChannelFactoryInterface $logger_factory, ConfigFactoryInterface $config_factory) {
    $this->webdamService = $webdam_service;
    $this->entityTypeManager = $entity_type_manager;
    $this->loggerFactory = $logger_factory;
    $this->configFactory = $config_factory;
  }

  /**
   * Generates hash based on active WebDAM config and saves it into state.
   *
   * @param \Drupal\amazon_sns\Event\SnsMessageEvent $event
   *   The Event to process.
   */
  public function onNotification(SnsMessageEvent $event) {
    if ($sns_message = $event->getMessage()) {

      // If the topic is not configured or not matching, do not process the
      // notification.
      $config = $this->configFactory->get('webdam_sns.settings');
      if (!$config->get('topic') || ($config->get('topic') != $sns_message['TopicArn'])) {
        return;
      }
      $message = Json::decode($sns_message['Message']);

      if (empty($message['assetId'])) {
        $this->loggerFactory->get('webdam_sns')->error('Notification not processed, no media ID found: @message', ['@message' => print_r($event->getMessage()->toArray(), TRUE)]);
        return;
      }

      $source_fields = [];
      foreach ($this->webdamService->getWebdamMediaTypes() as $media_type) {
        $source_fields[] = $media_type->getSource()->getConfiguration()['source_field'];
      }
      $source_fields = array_filter(array_unique($source_fields));
      if (empty($source_fields)) {
        $this->loggerFactory->get('webdam_sns')->error('Notification not processed, no Webdam source fields found.');
        return;
      }

      $storage = $this->entityTypeManager->getStorage('media');
      $query = $storage->getQuery();

      $source_field_condition = $query->orConditionGroup();
      foreach (array_unique($source_fields) as $source_field) {
        $source_field_condition->condition($source_field, $message['assetId']);
      }

      $mid = $query
        ->condition($source_field_condition)
        ->range(0, 1)
        ->accessCheck(FALSE)
        ->execute();

      if ($mid) {
        $media = $storage->load(reset($mid));
        $webdam_media_entities = [
          $message['assetId'] => $media,
        ];
        $updated = $this->webdamService->updateMediaEntities($webdam_media_entities);
        if ($updated) {
          $this->loggerFactory->get('webdam_sns')->notice('Updated webdam asset @label (@media_id)', [
            'link' => $media->toLink($this->t('View'))->toString(),
            '@media_id' => $message['assetId'],
            '@label' => $media->label(),
          ]);
        }
      }
      else {
        $this->loggerFactory->get('webdam_sns')->notice('Notification not processed, no media entity for ID @id found', ['@id' => $message['assetId']]);
        return;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[SnsEvents::NOTIFICATION][] = ['onNotification'];
    return $events;
  }

}
