<?php

namespace Drupal\webdam\Form;

use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\webdam\WebdamConnector;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\webdam\WebdamService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Webdam.
 */
class WebdamConfigurationForm extends ConfigFormBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Webdam client.
   *
   * @var \Drupal\webdam\WebdamConnector
   */
  protected $webdam;

  /**
   * @var \Drupal\webdam\WebdamService
   */
  protected WebdamService $webdamService;

  /**
   * Constructs a WebdamConfigurationForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\webdam\WebdamConnector $webdam
   *   The webdam client.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, RendererInterface $renderer, ModuleHandlerInterface $module_handler, WebdamConnector $webdam, WebdamService $webdam_service) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->renderer = $renderer;
    $this->moduleHandler = $module_handler;
    $this->webdam = $webdam;
    $this->webdamService = $webdam_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('renderer'),
      $container->get('module_handler'),
      $container->get('webdam.connector'),
      $container->get('webdam.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'webdam_configuration_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['webdam.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('webdam.settings');

    $form['credentials'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['credentials-wrapper']],
    ];

    $form['credentials']['account_domain'] = [
      '#required' => TRUE,
      '#type' => 'textfield',
      '#title' => $this->t('Account domain'),
      '#default_value' => $config->get('account_domain'),
      '#parents' => ['account_domain'],
      '#description' => $this->t('Provide your Webdam account domain, typically [name].webdamdb.com.'),
    ];

    $form['credentials']['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client ID'),
      '#parents' => ['credentials', 'client_id'],
      '#default_value' => $config->get('client_id'),
    ];
    $form['credentials']['client_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client secret'),
      '#parents' => ['credentials', 'client_secret'],
      '#default_value' => $config->get('client_secret'),
    ];
    $form['credentials']['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username'),
      '#parents' => ['credentials', 'username'],
      '#default_value' => $config->get('username'),

    ];
    $form['credentials']['password'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Password'),
      '#parents' => ['credentials', 'password'],
    ];
    if ($config->get('password')) {
      $form['credentials']['password']['#description'] = $this->t('The current password is not displayed. Enter a new password to replace it.');
    }
    $form['test'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('API connection test'),
      'wrapper' => [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#attributes' => ['id' => 'connection-test'],
        '#attached' => ['library' => ['webdam/config_form']],
      ],
      'check' => [
        '#type' => 'button',
        '#limit_validation_errors' => [],
        '#value' => $this->t('Test connection'),
        '#ajax' => ['callback' => '::testConnectionAjaxCallback'],
      ],
      'test_connection' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Test connection before saving'),
        '#description' => $this->t("Uncheck to allow saving credentials even if connection to Webdam can't be established."),
        '#default_value' => TRUE,
      ],
    ];

    $form['metadata'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Local metadata'),
      '#description' => $this->t('Updates the Webdam media entities metadata information in a batch process.'),
    ];
    $form['metadata']['update_frequency'] = [
      '#type' => 'number',
      '#step' => 1,
      '#min' => 0,
      '#title' => $this->t('Metadata update frequency'),
      '#default_value' => $config->get('update_frequency'),
      '#field_suffix' => $this->t('seconds'),
      '#description' => $this->t('The update frequency to update local metadata information in seconds.'),
    ];
    $form['metadata']['batch_update'] = [
      '#type' => 'submit',
      '#value' => $this->t('Update local metadata'),
      '#submit' => [[static::class, 'submitBatchMetadataUpdate']],
    ];

    $form['performance'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Performance settings'),
    ];

    $form['performance']['use_remote_images'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use remote image thumbnails of media and do not download them locally'),
      '#description' => $this->t('This can only be enabled if the <a href="@url">Remote Stream Wrapper module</a> is installed.', ['@url' => 'https://www.drupal.org/project/remote_stream_wrapper']),
      '#default_value' => $config->get('use_remote_images'),
      '#disabled' => !$this->moduleHandler->moduleExists('remote_stream_wrapper'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback for test connection button.
   */
  public function testConnectionAjaxCallback(array $form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $return_markup = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => ['id' => 'connection-test'],
    ];

    $credentials = $form_state->getValue('credentials');
    if (empty($credentials['password']) && !empty($this->config('webdam.settings')->get('password'))) {
      $credentials['password'] = $this->config('webdam.settings')->get('password');
    }

    $url = $form_state->getValue('account_domain');
    $has_credentials = isset(
      $credentials['username'],
      $credentials['password'],
      $credentials['client_id'],
      $credentials['client_secret'],
      $url
    );

    if ($has_credentials && $this->testApiConnection($credentials['username'], $credentials['password'], $credentials['client_id'], $credentials['client_secret'], $url)) {
      $return_markup['#value'] = $this->t('The API connection was established successfully.');
      $return_markup['#attributes']['style'] = 'color: green;';
    }
    else {
      $return_markup['#value'] = $this->t('Could not establish connection with Webdam. Check your credentials or <a href=":support">contact support.</a>', [':support' => 'mailto:support@getwebdam.com']);
      $return_markup['#attributes']['style'] = 'color: red;';
    }

    $response->addCommand(new ReplaceCommand('#connection-test', $this->renderer->render($return_markup)));
    return $response;
  }

  /**
   * Submit callback to trigger the metadata batch update.
   */
  public static function submitBatchMetadataUpdate(array &$form, FormStateInterface $form_state) {
    // Create a batch to process metadata of the Webdam media entities.
    $batch = [
      'operations' => [[WebdamConfigurationForm::class . '::updateMetadataBatchWorker', []]],
      'finished' => WebdamConfigurationForm::class . '::batchFinished',
      'title' => t('Updating local metadata...'),
      'error_message' => t('The update process has encountered an error.'),
    ];
    batch_set($batch);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $credentials = $form_state->getValue('credentials');

    if (empty($credentials['password']) && !empty($this->config('webdam.settings')->get('password'))) {
      $credentials['password'] = $this->config('webdam.settings')->get('password');
    }

    // Makes sure we don't have a leading slash in the domain url.
    $account_domain = $form_state->getValue('account_domain');
    if (!UrlHelper::isValid($account_domain, TRUE)) {
      $form_state->setErrorByName('account_domain', $this->t('This url is not valid. Make sure to include https://'));
    }

    $form_state->setValue('account_domain', rtrim($account_domain, '/'));

    if ($form_state->getValue('test_connection')) {
      $url = $form_state->getValue('account_domain');
      $has_credentials = isset(
        $credentials['username'],
        $credentials['password'],
        $credentials['client_id'],
        $credentials['client_secret'],
        $url
      );

      if (!$form_state::hasAnyErrors() && $has_credentials && !$this->testApiConnection($credentials['username'], $credentials['password'], $credentials['client_id'], $credentials['client_secret'], $url)) {
        $form_state->setErrorByName('credentials', $this->t('Could not establish connection with Webdam. Check your credentials or <a href=":support">contact support.</a>', [':support' => 'mailto:support@getwebdam.com']));
      }
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $credentials = $form_state->getValue('credentials');
    $config = $this->config('webdam.settings');
    $config
      ->set('account_domain', $form_state->getValue('account_domain'))
      ->set('update_frequency', $form_state->getValue('update_frequency'));

    foreach ($credentials as $name => $credential) {
      if (!empty($credential)) {
        $config->set($name, trim($credential));
      }
    }

    $config->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Batch operation worker to update the metadata of Webdam media entities.
   *
   * @param array $context
   *   The context array.
   */
  public static function updateMetadataBatchWorker(array &$context) {
    /** @var \Drupal\webdam\WebdamService $webdam */
    $webdam = \Drupal::service('webdam.service');

    if (empty($context['sandbox']['total'])) {
      $context['sandbox']['progress'] = 0;
      $context['sandbox']['current_id'] = -1;
      $context['sandbox']['total'] = $webdam->getTotalCountOfMediaEntities();
      $context['results']['updated'] = 0;
      $context['results']['skipped'] = 0;
    }

    $minimum_id = $context['sandbox']['current_id'] > 0 ? $context['sandbox']['current_id'] : NULL;

    $results = $webdam->updateMetadataLastMediaEntities($minimum_id);

    if (empty($results)) {
      $context['finished'] = 1;
      $context['sandbox']['progress'] = $context['sandbox']['total'];
      $context['results'] = [];
      return;
    }

    $context['sandbox']['current_id'] = $results['max_id'];
    $context['sandbox']['progress'] = $context['sandbox']['progress'] + $results['total'];
    $context['results']['updated'] = $context['results']['updated'] + count($results['updated']);
    $context['results']['skipped'] = $context['results']['skipped'] + count($results['skipped']);

    if ($context['sandbox']['progress'] < $context['sandbox']['total']) {
      $context['finished'] = $context['sandbox']['progress'] / $context['sandbox']['total'];
    }
    else {
      $context['finished'] = 1;
    }

    $context['message'] = t('Updating local metadata @current of @total', [
      '@current' => $context['sandbox']['progress'],
      '@total' => $context['sandbox']['total'],
    ]);
  }

  /**
   * Finish callback for the batch processing.
   *
   * @param bool $success
   *   Whether the batch completed successfully.
   * @param array $results
   *   The results array.
   * @param array $operations
   *   The operations array.
   */
  public static function batchFinished($success, array $results, array $operations) {
    $messenger = \Drupal::messenger();
    if (!empty($results)) {
      $messenger->addStatus(t('Updated local metadata for @count media entities.', ['@count' => $results['updated']]));
      if ($results['skipped']) {
        $messenger->addWarning(t('Failed to update the local metadata for @count media entities missing in the remote system.', ['@count' => $results['skipped']]));
      }
    }
    else {
      $messenger->addWarning(t('There is an error with the connection to the Webdam service or there are no Webdam media entities to update.'));
    }
  }

  /**
   * Tests connection with the Webdam api.
   *
   * @param string $username
   *   The username.
   * @param string $password
   *   The password.
   * @param string $client_id
   *   Client id.
   * @param string $client_secret
   *   Client secret.
   * @param string $url
   *   Url.
   *
   * @return bool
   *   Whether communication was successfully established.
   */
  protected function testApiConnection(string $username, string $password, string $client_id, string $client_secret, string $url) {
    try {
      $client = $this->webdam->doGetClient($username, $password, $client_id, $client_secret);
      $client->checkAuth();
    }
    catch (\Exception $exception) {
      return FALSE;
    }
    return TRUE;
  }

}
