<?php

namespace Drupal\webdam\Plugin\EntityBrowser\Widget;

use Drupal\webdam\WebdamConnector;
use Drupal\webdam\Exception\BundleNotWebdamException;
use Drupal\webdam\Exception\BundleNotExistException;
use Drupal\webdam\Exception\UnableToConnectException;
use Drupal\webdam\Plugin\media\Source\Webdam;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\entity_browser\WidgetBase;
use Drupal\entity_browser\WidgetValidationManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Base class for Webdam Entity browser widgets.
 */
abstract class WebdamWidgetBase extends WidgetBase {

  /**
   * Webdam connector.
   *
   * @var \Drupal\webdam\WebdamConnector
   */
  protected $webdamConnector;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactory
   */
  protected $loggerFactory;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * WebdamWidgetBase constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   Event dispatcher service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\entity_browser\WidgetValidationManager $validation_manager
   *   The Widget Validation Manager service.
   * @param \Drupal\webdam\WebdamConnector $webdam
   *   The webdam conector.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   Logger factory.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EventDispatcherInterface $event_dispatcher, EntityTypeManagerInterface $entity_type_manager, WidgetValidationManager $validation_manager, WebdamConnector $webdam, LoggerChannelFactoryInterface $logger_factory, LanguageManagerInterface $language_manager, RequestStack $request_stack, ConfigFactoryInterface $config_factory) {
    $this->webdamConnector = $webdam;
    $this->loggerFactory = $logger_factory;
    $this->languageManager = $language_manager;
    $this->requestStack = $request_stack;
    $this->config = $config_factory;

    parent::__construct($configuration, $plugin_id, $plugin_definition, $event_dispatcher, $entity_type_manager, $validation_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('event_dispatcher'),
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.entity_browser.widget_validation'),
      $container->get('webdam.connector'),
      $container->get('logger.factory'),
      $container->get('language_manager'),
      $container->get('request_stack'),
      $container->get('config.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'media_type' => NULL,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['media_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Media type'),
      '#default_value' => $this->configuration['media_type'],
      '#required' => TRUE,
      '#options' => [],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getForm(array &$original_form, FormStateInterface $form_state, array $additional_widget_parameters) {
    $form = parent::getForm($original_form, $form_state, $additional_widget_parameters);

    if (!$this->checkType()) {
      $form_state->setValue('errors', TRUE);
      return $form;
    }

    // Check if the API configuration is in place and exit early if not.
    foreach (['client_id', 'client_secret', 'account_domain', 'username', 'password'] as $key) {
      if ($this->config->get('webdam.settings')->get($key) === '') {
        $form_state->setValue('errors', TRUE);
        (new UnableToConnectException())->logException()->displayMessage();
        return $form;
      }
    }

    return $form;
  }

  /**
   * Check that media type is properly configured.
   *
   * @return bool
   *   Returns TRUE if media type is configured correctly.
   */
  protected function checkType() {
    if ($this->configuration['media_type']) {
      /** @var \Drupal\media\MediaTypeInterface $type */
      $type = $this->entityTypeManager->getStorage('media_type')
        ->load($this->configuration['media_type']);

      if (!$type) {
        (new BundleNotExistException(
          $this->configuration['media_type']
        ))->logException()->displayMessage();
        return FALSE;
      }
      elseif (!($type->getSource() instanceof Webdam)) {
        (new BundleNotWebdamException($type->label()))->logException()
          ->displayMessage();
        return FALSE;
      }
      return TRUE;
    }
  }

}
