<?php

namespace Drupal\webdam\Plugin\Field\FieldFormatter;

use Drupal\webdam\Plugin\media\Source\Webdam;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Template\Attribute;

/**
 * Plugin implementation of the 'Webdam Video' formatter.
 *
 * @FieldFormatter(
 *   id = "webdam_video",
 *   label = @Translation("Webdam (Video)"),
 *   field_types = {"string", "string_long"}
 * )
 */
class WebdamVideoFormatter extends WebdamFormatterBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'controls' => TRUE,
      'autoplay' => FALSE,
      'loop' => FALSE,
      'muted' => FALSE,
      'width' => 640,
      'height' => 480,
    ] + parent::defaultSettings();
  }
  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);
    $elements['controls'] = [
      '#title' => $this->t('Show playback controls'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('controls'),
    ];
    $elements['autoplay'] = [
      '#title' => $this->t('Autoplay'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('autoplay'),
    ];
    $elements['loop'] = [
      '#title' => $this->t('Loop'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('loop'),
    ];
    $elements['muted'] = [
      '#title' => $this->t('Muted'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('muted'),
    ];
    $elements['width'] = [
      '#type' => 'number',
      '#title' => $this->t('Width'),
      '#default_value' => $this->getSetting('width'),
      '#size' => 5,
      '#maxlength' => 5,
      '#field_suffix' => $this->t('pixels'),
      '#min' => 0,
      '#required' => TRUE,
    ];
    $elements['height'] = [
      '#type' => 'number',
      '#title' => $this->t('Height'),
      '#default_value' => $this->getSetting('height'),
      '#size' => 5,
      '#maxlength' => 5,
      '#field_suffix' => $this->t('pixels'),
      '#min' => 0,
      '#required' => TRUE,
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    $summary[] = $this->t('Controls: %controls', ['%controls' => $this->getSetting('controls') ? $this->t('yes') : $this->t('no')]);
    $summary[] = $this->t('Autoplay: %autoplay', ['%autoplay' => $this->getSetting('autoplay') ? $this->t('yes') : $this->t('no')]);
    $summary[] = $this->t('Loop: %loop', ['%loop' => $this->getSetting('loop') ? $this->t('yes') : $this->t('no')]);
    $summary[] = $this->t('Muted: %muted', ['%muted' => $this->getSetting('muted') ? $this->t('yes') : $this->t('no')]);
    $summary[] = $this->t('Size: %width x %height pixels', [
      '%width' => $this->getSetting('width'),
      '%height' => $this->getSetting('height'),
    ]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {

    $elements = [];
    $is_entityreference = $this->fieldDefinition->getType() == 'entity_reference';

    foreach ($items as $delta => $item) {

      /** @var \Drupal\media\MediaInterface $media */
      $media = $is_entityreference ? $item->entity : $items->getEntity();
      if (!$media) {
        continue;
      }
      $source_plugin = $media->getSource();
      if ($source_plugin instanceof Webdam && $video_url = $source_plugin->getMetadata($media, 'video_url')) {
        $attributes = new Attribute();
        $attributes->setAttribute('controls', $this->getSetting('controls'))
          ->setAttribute('autoplay', $this->getSetting('autoplay'))
          ->setAttribute('loop', $this->getSetting('loop'))
          ->setAttribute('mute', $this->getSetting('mute'))
          ->setAttribute('width', $this->getSetting('width'))
          ->setAttribute('height', $this->getSetting('height'));

        $video_preview_urls = $source_plugin->getMetadata($media, 'video_preview_urls');
        if ($video_preview_urls) {
          $attributes->setAttribute('poster', reset($video_preview_urls));
        }

        $source_attribute = new Attribute();
        $source_attribute->setAttribute('src', $video_url);
        $extension_type = $this->guessType($media);
        $source_attribute->setAttribute('type', "video/$extension_type");

        $elements[] = [
          '#theme' => 'webdam_video',
          '#attributes' => $attributes,
          '#source_attributes' => [$source_attribute],
        ];
      }
    }

    return $elements;
  }

  /**
   * Guess the type from the extension.
   *
   * Try to get the extension from metadata.
   * If unable to, get if from the filename.
   * Defaults to mp4.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   *
   * @return string
   *   The type string.
   */
  protected function guessType($media) {
    $source_plugin = $media->getSource();
    $video_url = $source_plugin->getMetadata($media, 'video_url');

    $extension_type = 'mp4';
    $path_info = pathinfo($video_url);
    if ($path_info['extension']) {
      $extension_type = substr($path_info['extension'], 0, strpos($path_info['extension'], '?'));
    }

    return $extension_type;
  }

}
